----------------------------------------------------------------------------------------------
--
-- The Ur-Creature Generator
-- =========================
--
-- by Tony Jones, Version 1.0, August 2003
--
-- This program generates an Ur-creature, ready for uplift, using the system given in
-- chapter 4 of GURPS Uplift, first edition. GURPS is copyright Steve Jackson Games while
-- Uplift and the Uplift universe are the property of the author, David Brin. As such
-- concepts from them both are used here, without permission, but for game use only and
-- not for profit or commercial gain.
--
-- For more, see http://www.sjgames.com/gurps/books/uplift/.
--
----------------------------------------------------------------------------------------------
--
-- To use this program simply double-click on the executable. The program will then ask you
-- a name for your creature. If you just press return then the name will default to Ur_Creature.
-- The program will then write the details of the creature out to a file named
-- <creature name>.txt (if no name is specified then this will be Ur_Creature.txt). Note that
-- the program does not check for existing files, and will overwrite an existing file without
-- telling you, so it's best to move or rename any ur-creatures you particularly like!
--
----------------------------------------------------------------------------------------------
--
-- This program is written in Ada 95. The Ada 95 compiler which was used is the free GNAT Ada
-- compiler, running with the AdaGIDE GUI front-end. The GNAT compiler can be downloaded from
-- http://libra.act-europe.fr/GNAT/main.html, while AdaGIDE can be downloaded from
-- http://www.usafa.af.mil/dfcs/bios/mcc_html/adagide.html.
--
----------------------------------------------------------------------------------------------
--
-- Copyright (c) Tony Jones 2005
--
-- This work is licensed under a Creative Commons Attribution-NonCommercial-ShareAlike 2.5
-- Licence (see http://creativecommons.org/licenses/by-nc-sa/2.5/)
--
-- You are free to copy, distribute, display, and perform the work, and to make derivative
-- works, under the following conditions:
--
-- * Attribution. You must give the original author credit.
-- * Non-Commercial. You may not use this work for commercial purposes.
-- * Share Alike. If you alter, transform, or build upon this work, you may distribute the
--   resulting work only under a licence identical to this one.
-- * For any reuse or distribution, you must make clear to others the licence terms of
--   this work.
--
-- Any of these conditions can be waived if you get permission from the copyright holder,
-- who can be contacted via his website at http://www.wolfram.demon.co.uk/.
--
-- Your fair use and other rights are in no way affected by the above.
--
-- This is a human-readable summary of the Legal Code (the full licence), available at
-- http://creativecommons.org/licenses/by-nc-sa/2.5/legalcode.
--
----------------------------------------------------------------------------------------------

with Ada.Characters.Handling;
with Ada.Characters.Latin_1;
with Ada.Numerics.Discrete_Random;
with Ada.Strings.Fixed;
with Ada.Strings.Unbounded;
with Ada.Text_Io;

procedure Create_Ur_Species is

  --------------------------------------------------------------------------------------------
  -- Types
  subtype D_6_Type is integer range 1 .. 6;

  type Environment_Type is (Desert, Tundra_Steppe, Grassland, Forest, Seashore, Unusual, None);
  type Unusual_Env_Type is (Ice_Desert, Mountains_Cliff, Burrower, Swamp_Bog, Aquatic, Exotic, None);
  type Diet_Type        is (Carnivore, Omnivore, Herbivore, Ergivore);
  type Feeding_Type     is (Stalker, Chaser, Scavenger, Pouncer, Herder, Trapper, -- Carnivore type
                            Gatherer_Hunter, Opportunist_Browser, Hunter_Browser, -- Omnivore types
                            Grazer, Browser, Saprophytic, Gatherer,               -- Herbivore types
                            Pure, Mixed, Rooted, Tapper);                         -- Ergivore types

  type Metabolism_Type  is (Cold_Blooded, Warm_Blooded, Hyperactive);
  type Society_Type     is (Solitary, Pair_Bonding, Family_Group, Pack_Troop, Herd, Hive);
  type Size_Type        is (Tiny, Small, Short, Average, Tall, Big, Huge, Giant);
  type Bulk_Type        is (Slight, Light, Average, Heavy, Bulky);
  type Chain_Posn_Type  is (Low, Middle, Near_Top, Top_Shared, Top_Solitary);
  type Offspring_Type   is (Hordes_Very_Little, Many_Low, Several_Moderate, Few_Much, Very_Few_Very_Much);
  type Symmetry_Type    is (Bilateral, Radial);
  type Posture_Type     is (Horizontal, Centauroid, Semi_Upright, Upright, Invertebrate);
  type Kinesthetic_Type is (None, Poor, Average, Good, Excellent);

  type Others_Type      is array (1 .. 100) of Ada.Strings.Unbounded.Unbounded_String;
  type Repeat_Data_Type is array (1 ..   3) of Ada.Strings.Unbounded.Unbounded_String;
  type Sc_Type          is array (1 ..   3) of Natural;
  
  type Creature_Data_Type is record
    Name            : Ada.Strings.Unbounded.Unbounded_String;
    Environment     : Environment_Type;
    Secondary_Env   : Environment_Type;
    Unusual_Env     : Unusual_Env_Type;
    Diet            : Diet_Type;
    Feeding_Method  : Feeding_Type;
    Secondary_Diet  : Diet_Type;
    Sec_Feed_Meth   : Feeding_Type;
    Metabolism      : Metabolism_Type;
    Society         : Society_Type;
    Size            : Size_Type;
    Avg_Weight      : Natural;
    Bulk            : Bulk_Type;
    Brachiator      : Boolean;
    Swimmer         : Boolean;
    Fully_Aquatic   : Boolean;
    Amphibious      : Boolean;
    Glider          : Boolean;
    Winged_Flight   : Boolean;
    Tunneller       : Boolean;
    Bog_Dweller     : Boolean;
    Incr_Strength   : Boolean;
    Chauvinism      : Integer;
    Concentration   : Integer;
    Curiosity       : Integer;
    Egotism         : Integer;
    Empathy         : Integer;
    Gregariousness  : Integer;
    Imagination     : Integer;
    Suspicion       : Integer;
    Chauv_Trait     : Ada.Strings.Unbounded.Unbounded_String;
    Conc_Trait      : Ada.Strings.Unbounded.Unbounded_String;
    Curio_Trait     : Ada.Strings.Unbounded.Unbounded_String;
    Ego_Trait       : Ada.Strings.Unbounded.Unbounded_String;
    Emp_Trait       : Ada.Strings.Unbounded.Unbounded_String;
    Gregar_Trait    : Ada.Strings.Unbounded.Unbounded_String;
    Imag_Trait      : Ada.Strings.Unbounded.Unbounded_String;
    Susp_Trait      : Ada.Strings.Unbounded.Unbounded_String;
    Dx              : Natural;
    Ht              : Natural;
    Hits            : Natural;
    St              : Natural;
    Iq              : Natural;
    Num_Other       : Natural;
    Other           : Others_Type;
    Chain_Posn      : Chain_Posn_Type;
    Offspring       : Offspring_Type; 
    Lifespan        : Ada.Strings.Unbounded.Unbounded_String;
    Genders         : Ada.Strings.Unbounded.Unbounded_String;
    Delivery        : Ada.Strings.Unbounded.Unbounded_String;
    Reprod_Spec     : Repeat_Data_Type;
    Individ_Queens  : Boolean;
    Dominant_Sex    : Boolean;
    Casted          : Boolean;
    Symmetry        : Symmetry_Type;
    Num_Limbs       : Natural;
    Walking_Limbs   : Natural;
    Walk_Limb_Type  : Ada.Strings.Unbounded.Unbounded_String;
    Handling_Limbs  : Natural;
    Hand_Limb_Type  : Ada.Strings.Unbounded.Unbounded_String;
    Num_Extra_Limbs : Natural;
    Ext_Limb_St     : Ada.Strings.Unbounded.Unbounded_String;
    Ext_Limb_Dx     : Ada.Strings.Unbounded.Unbounded_String;
    Ext_Limb_Sens   : Ada.Strings.Unbounded.Unbounded_String;
    Posture         : Posture_Type;
    Natural_Weapons : Repeat_Data_Type;
    Natural_Poisons : Repeat_Data_Type;
    Poison_Potency  : Natural;
    Extrord_Weapons : Ada.Strings.Unbounded.Unbounded_String;
    Integument      : Ada.Strings.Unbounded.Unbounded_String;
    Voice_Box       : Ada.Strings.Unbounded.Unbounded_String;
    Special_Comms   : Ada.Strings.Unbounded.Unbounded_String;
    Vision          : Ada.Strings.Unbounded.Unbounded_String;
    Vision_Done     : Boolean;
    Blind           : Boolean;
    Number_Of_Eyes  : Natural;
    Eye_Placement   : Ada.Strings.Unbounded.Unbounded_String;
    Special_Vision  : Repeat_Data_Type;
    Hearing         : Ada.Strings.Unbounded.Unbounded_String;
    Special_Hearing : Repeat_Data_Type;
    Kinesthetic     : Ada.Strings.Unbounded.Unbounded_String;
    Kinesthetic_2   : Kinesthetic_Type;
    Special_Kines   : Ada.Strings.Unbounded.Unbounded_String;
    Taste           : Ada.Strings.Unbounded.Unbounded_String;
    Smell           : Ada.Strings.Unbounded.Unbounded_String;
    Presentient     : Boolean;
    Special_Ability : Ada.Strings.Unbounded.Unbounded_String;
    Social_Standing : Ada.Strings.Unbounded.Unbounded_String;
    Special_Cases   : Repeat_Data_Type;
    Wealth          : Ada.Strings.Unbounded.Unbounded_String;
  end record;

  --------------------------------------------------------------------------------------------
  -- Constants
  Null_Repeat_Data : constant Repeat_Data_Type :=
    (Others => Ada.Strings.Unbounded.Null_Unbounded_String);
  
  Null_Other_Data  : constant Others_Type :=
    (Others => Ada.Strings.Unbounded.Null_Unbounded_String);
  
  Null_Creature    : constant Creature_Data_Type :=
    (Name            => Ada.Strings.Unbounded.To_Unbounded_String ("Ur_Species"),
     Environment     => None,
     Secondary_Env   => None,
     Unusual_Env     => None,
     Diet            => Carnivore,
     Feeding_Method  => Grazer,
     Secondary_Diet  => Ergivore,
     Sec_Feed_Meth   => Grazer,
     Metabolism      => Cold_Blooded,
     Society         => Solitary,
     Size            => Tiny,
     Avg_Weight      => 999,
     Bulk            => Bulky,
     Brachiator      => False,
     Swimmer         => False,
     Fully_Aquatic   => False,
     Amphibious      => False,
     Glider          => False,
     Winged_Flight   => False,
     Tunneller       => False,
     Bog_Dweller     => False,
     Incr_Strength   => False,
     Chauvinism      => 0,
     Concentration   => 0,
     Curiosity       => 0,
     Egotism         => 0,
     Empathy         => 0,
     Gregariousness  => 0,
     Imagination     => 0,
     Suspicion       => 0,
     Chauv_Trait     => Ada.Strings.Unbounded.Null_Unbounded_String,
     Conc_Trait      => Ada.Strings.Unbounded.Null_Unbounded_String,
     Curio_Trait     => Ada.Strings.Unbounded.Null_Unbounded_String,
     Ego_Trait       => Ada.Strings.Unbounded.Null_Unbounded_String,
     Emp_Trait       => Ada.Strings.Unbounded.Null_Unbounded_String,
     Gregar_Trait    => Ada.Strings.Unbounded.Null_Unbounded_String,
     Imag_Trait      => Ada.Strings.Unbounded.Null_Unbounded_String,
     Susp_Trait      => Ada.Strings.Unbounded.Null_Unbounded_String,
     Dx              => 10,
     Ht              => 10,
     Hits            => 10,
     St              => 10,
     Iq              => 0,
     Num_Other       => 0,
     Other           => Null_Other_Data,
     Chain_Posn      => Low,
     Offspring       => Hordes_Very_Little,
     Lifespan        => Ada.Strings.Unbounded.Null_Unbounded_String,
     Genders         => Ada.Strings.Unbounded.Null_Unbounded_String,
     Delivery        => Ada.Strings.Unbounded.Null_Unbounded_String,
     Reprod_Spec     => Null_Repeat_Data,
     Individ_Queens  => False,
     Dominant_Sex    => False,
     Casted          => False,
     Symmetry        => Bilateral,
     Num_Limbs       => 0,
     Walking_Limbs   => 0,
     Walk_Limb_Type  => Ada.Strings.Unbounded.Null_Unbounded_String,
     Handling_Limbs  => 0,
     Hand_Limb_Type  => Ada.Strings.Unbounded.Null_Unbounded_String,
     Num_Extra_Limbs => 0,
     Ext_Limb_St     => Ada.Strings.Unbounded.Null_Unbounded_String,
     Ext_Limb_Dx     => Ada.Strings.Unbounded.Null_Unbounded_String,
     Ext_Limb_Sens   => Ada.Strings.Unbounded.Null_Unbounded_String,
     Posture         => Invertebrate,
     Natural_Weapons => Null_Repeat_Data,
     Natural_Poisons => Null_Repeat_Data,
     Poison_Potency  => 0,
     Extrord_Weapons => Ada.Strings.Unbounded.Null_Unbounded_String,
     Integument      => Ada.Strings.Unbounded.Null_Unbounded_String,
     Voice_Box       => Ada.Strings.Unbounded.Null_Unbounded_String,
     Special_Comms   => Ada.Strings.Unbounded.Null_Unbounded_String,
     Vision          => Ada.Strings.Unbounded.Null_Unbounded_String,
     Vision_Done     => False,
     Blind           => False,
     Number_Of_Eyes  => 0,
     Eye_Placement   => Ada.Strings.Unbounded.Null_Unbounded_String,
     Special_Vision  => Null_Repeat_Data,
     Hearing         => Ada.Strings.Unbounded.Null_Unbounded_String,
     Special_Hearing => Null_Repeat_Data,
     Kinesthetic     => Ada.Strings.Unbounded.Null_Unbounded_String,
     Kinesthetic_2   => None,
     Special_Kines   => Ada.Strings.Unbounded.Null_Unbounded_String,
     Taste           => Ada.Strings.Unbounded.Null_Unbounded_String,
     Smell           => Ada.Strings.Unbounded.Null_Unbounded_String,
     Presentient     => False,
     Special_Ability => Ada.Strings.Unbounded.Null_Unbounded_String,
     Social_Standing => Ada.Strings.Unbounded.Null_Unbounded_String,
     Special_Cases   => Null_Repeat_Data,
     Wealth          => Ada.Strings.Unbounded.Null_Unbounded_String
    );

  -- All of the following constants are used when outputting the Ur-creature
  Environments : constant array (Environment_Type) of Ada.Strings.Unbounded.Unbounded_String := 
    (Desert        => Ada.Strings.Unbounded.To_Unbounded_String ("Desert"),
     Tundra_Steppe => Ada.Strings.Unbounded.To_Unbounded_String ("Tundra or Steppe"), 
     Grassland     => Ada.Strings.Unbounded.To_Unbounded_String ("Grassland or other Open Country"), 
     Forest        => Ada.Strings.Unbounded.To_Unbounded_String ("Forest"), 
     Seashore      => Ada.Strings.Unbounded.To_Unbounded_String ("Seashore"), 
     Unusual       => Ada.Strings.Unbounded.Null_Unbounded_String, 
     None          => Ada.Strings.Unbounded.Null_Unbounded_String);
     
  Unusual_Environments : constant array (Unusual_Env_Type) of Ada.Strings.Unbounded.Unbounded_String := 
    (Ice_Desert      => Ada.Strings.Unbounded.To_Unbounded_String ("Ice Desert"),
     Mountains_Cliff => Ada.Strings.Unbounded.To_Unbounded_String ("Mountains or Cliffside"), 
     Burrower        => Ada.Strings.Unbounded.To_Unbounded_String ("Burrower"), 
     Swamp_Bog       => Ada.Strings.Unbounded.To_Unbounded_String ("Swamp or Bog"), 
     Aquatic         => Ada.Strings.Unbounded.To_Unbounded_String ("Fully Aquatic"), 
     Exotic          => Ada.Strings.Unbounded.To_Unbounded_String ("Exotic Environment"), 
     None            => Ada.Strings.Unbounded.Null_Unbounded_String);
     
  Diets : constant array (Diet_Type) of Ada.Strings.Unbounded.Unbounded_String := 
    (Carnivore => Ada.Strings.Unbounded.To_Unbounded_String ("Carnivore"),
     Omnivore  => Ada.Strings.Unbounded.To_Unbounded_String ("Omnivore"), 
     Herbivore => Ada.Strings.Unbounded.To_Unbounded_String ("Herbivore"), 
     Ergivore  => Ada.Strings.Unbounded.To_Unbounded_String ("Ergivore"));
     
  Feeding_Methods : constant array (Feeding_Type) of Ada.Strings.Unbounded.Unbounded_String := 
    (Stalker             => Ada.Strings.Unbounded.To_Unbounded_String ("Stalker"),
     Chaser              => Ada.Strings.Unbounded.To_Unbounded_String ("Chaser"), 
     Scavenger           => Ada.Strings.Unbounded.To_Unbounded_String ("Scavenger"), 
     Pouncer             => Ada.Strings.Unbounded.To_Unbounded_String ("Pouncer"), 
     Herder              => Ada.Strings.Unbounded.To_Unbounded_String ("Herder"), 
     Trapper             => Ada.Strings.Unbounded.To_Unbounded_String ("Trapper"), 
     Gatherer_Hunter     => Ada.Strings.Unbounded.To_Unbounded_String ("Gatherer/Hunter"), 
     Opportunist_Browser => Ada.Strings.Unbounded.To_Unbounded_String ("Opportunist Browser"), 
     Hunter_Browser      => Ada.Strings.Unbounded.To_Unbounded_String ("Hunter/Browser"), 
     Grazer              => Ada.Strings.Unbounded.To_Unbounded_String ("Grazer"), 
     Browser             => Ada.Strings.Unbounded.To_Unbounded_String ("Browser"), 
     Saprophytic         => Ada.Strings.Unbounded.To_Unbounded_String ("Saprophytic"), 
     Gatherer            => Ada.Strings.Unbounded.To_Unbounded_String ("Gatherer"), 
     Pure                => Ada.Strings.Unbounded.To_Unbounded_String ("Pure"), 
     Mixed               => Ada.Strings.Unbounded.To_Unbounded_String ("Mixed"), 
     Rooted              => Ada.Strings.Unbounded.To_Unbounded_String ("Rooted"), 
     Tapper              => Ada.Strings.Unbounded.To_Unbounded_String ("Tapper"));
     
  Metabolisms : constant array (Metabolism_Type) of Ada.Strings.Unbounded.Unbounded_String := 
    (Cold_Blooded => Ada.Strings.Unbounded.To_Unbounded_String ("Cold-Blooded"),
     Warm_Blooded => Ada.Strings.Unbounded.To_Unbounded_String ("Warm-Blooded"), 
     Hyperactive  => Ada.Strings.Unbounded.To_Unbounded_String ("Hyperactive"));
     
  Societies : constant array (Society_Type) of Ada.Strings.Unbounded.Unbounded_String := 
    (Solitary     => Ada.Strings.Unbounded.To_Unbounded_String ("Solitary"),
     Pair_Bonding => Ada.Strings.Unbounded.To_Unbounded_String ("Pair-Bonding"), 
     Family_Group => Ada.Strings.Unbounded.To_Unbounded_String ("Family Group"), 
     Pack_Troop   => Ada.Strings.Unbounded.To_Unbounded_String ("Pack or Troop"), 
     Herd         => Ada.Strings.Unbounded.To_Unbounded_String ("Herd"),
     Hive         => Ada.Strings.Unbounded.To_Unbounded_String ("Hive"));
     
  Sizes : constant array (Size_Type) of Ada.Strings.Unbounded.Unbounded_String := 
    (Tiny    => Ada.Strings.Unbounded.To_Unbounded_String ("Tiny"),
     Small   => Ada.Strings.Unbounded.To_Unbounded_String ("Small"), 
     Short   => Ada.Strings.Unbounded.To_Unbounded_String ("Short"), 
     Average => Ada.Strings.Unbounded.To_Unbounded_String ("Average"), 
     Tall    => Ada.Strings.Unbounded.To_Unbounded_String ("Tall"),
     Big     => Ada.Strings.Unbounded.To_Unbounded_String ("Big"),
     Huge    => Ada.Strings.Unbounded.To_Unbounded_String ("Huge"),
     Giant   => Ada.Strings.Unbounded.To_Unbounded_String ("Giant"));
     
  Bulks : constant array (Bulk_Type) of Ada.Strings.Unbounded.Unbounded_String := 
    (Slight  => Ada.Strings.Unbounded.To_Unbounded_String ("Slight"),
     Light   => Ada.Strings.Unbounded.To_Unbounded_String ("Light"), 
     Average => Ada.Strings.Unbounded.To_Unbounded_String ("Average"), 
     Heavy   => Ada.Strings.Unbounded.To_Unbounded_String ("Heavy"), 
     Bulky   => Ada.Strings.Unbounded.To_Unbounded_String ("Bulky"));
     
  Chain_Positions : constant array (Chain_Posn_Type) of Ada.Strings.Unbounded.Unbounded_String := 
    (Low          => Ada.Strings.Unbounded.To_Unbounded_String ("Low"),
     Middle       => Ada.Strings.Unbounded.To_Unbounded_String ("Middle"), 
     Near_Top     => Ada.Strings.Unbounded.To_Unbounded_String ("Near Top"), 
     Top_Shared   => Ada.Strings.Unbounded.To_Unbounded_String ("Top, Shared"), 
     Top_Solitary => Ada.Strings.Unbounded.To_Unbounded_String ("Top, Solitary"));
     
  Offsprings : constant array (Offspring_Type) of Ada.Strings.Unbounded.Unbounded_String := 
    (Hordes_Very_Little => Ada.Strings.Unbounded.To_Unbounded_String ("Hordes/Very Little"),
     Many_Low           => Ada.Strings.Unbounded.To_Unbounded_String ("Many/Low"), 
     Several_Moderate   => Ada.Strings.Unbounded.To_Unbounded_String ("Several/Moderate"), 
     Few_Much           => Ada.Strings.Unbounded.To_Unbounded_String ("Few/Much"), 
     Very_Few_Very_Much => Ada.Strings.Unbounded.To_Unbounded_String ("Very Few/Very Much"));
     
  Symmetries : constant array (Symmetry_Type) of Ada.Strings.Unbounded.Unbounded_String := 
    (Bilateral => Ada.Strings.Unbounded.To_Unbounded_String ("Bilateral"),
     Radial    => Ada.Strings.Unbounded.To_Unbounded_String ("Radial"));
     
  Postures : constant array (Posture_Type) of Ada.Strings.Unbounded.Unbounded_String := 
    (Horizontal   => Ada.Strings.Unbounded.To_Unbounded_String ("Horizontal"),
     Centauroid   => Ada.Strings.Unbounded.To_Unbounded_String ("Centauroid"), 
     Semi_Upright => Ada.Strings.Unbounded.To_Unbounded_String ("Semi-Upright"), 
     Upright      => Ada.Strings.Unbounded.To_Unbounded_String ("Upright"), 
     Invertebrate => Ada.Strings.Unbounded.To_Unbounded_String ("Invertebrate"));
     
  --------------------------------------------------------------------------------------------
  -- Packages
  package Int_Io is new Ada.Text_Io.Integer_Io       (Natural);
  package D_6    is new Ada.Numerics.Discrete_Random (D_6_Type);
  
  --------------------------------------------------------------------------------------------
  -- Variables
  Br          : Sc_Type            := (Others => 0);
  Creature    : Creature_Data_Type := Null_Creature;
  Ctp         : Sc_Type            := (Others => 0);
  Dice        : D_6.Generator;
  Lmp         : Sc_Type            := (Others => 0);
  Mi          : Sc_Type            := (Others => 0);
  Output_File : Ada.Text_Io.File_Type;
  Total       : Integer;
  Spec_Num    : Integer            := 0;
  Sym_Number  : Natural;

  --------------------------------------------------------------------------------------------
  -- Constants
  Base_Weight : constant array (Size_Type) of Natural :=
    (Tiny    => 45,
     Small   => 80, 
     Short   => 135, 
     Average => 150, 
     Tall    => 180, 
     Big     => 210, 
     Huge    => 400, 
     Giant   => 900
    );
    
  Size_St_Mod : constant array (Size_Type) of Integer :=
    (Tiny    => -6,
     Small   => -3, 
     Short   => -1, 
     Average =>  0, 
     Tall    =>  1, 
     Big     =>  2, 
     Huge    =>  4, 
     Giant   =>  6
    );
    
  Size_Ht_Mod : constant array (Size_Type) of Integer :=
    (Tiny    => -2,
     Small   => -1, 
     Short   =>  0, 
     Average =>  0, 
     Tall    =>  1, 
     Big     =>  2, 
     Huge    =>  4, 
     Giant   =>  6
    );
    
  Size_Hits_Mod : constant array (Size_Type) of Integer :=
    (Tiny    => -4,
     Small   => -3, 
     Short   =>  0, 
     Average =>  0, 
     Tall    =>  1, 
     Big     =>  2, 
     Huge    =>  6, 
     Giant   =>  10
    );
    
  Bulk_Multiplier : constant array (Bulk_Type) of Float := 
    (Slight  => 0.6,
     Light   => 0.8, 
     Average => 1.0, 
     Heavy   => 1.2, 
     Bulky   => 1.5
    );

  Bulk_Ht_Mod : constant array (Bulk_Type) of Integer := 
    (Slight  => -2,
     Light   => -1, 
     Average =>  0, 
     Heavy   =>  1, 
     Bulky   =>  2
    );

  --------------------------------------------------------------------------------------------
  -- Uses
  use type Ada.Strings.Unbounded.Unbounded_String;

  --------------------------------------------------------------------------------------------
  -- Functions
  -- This simulates a d6
  function One_D return D_6_Type is
  begin
    return D_6.Random (Gen => Dice);
  end One_D;

  --------------------------------------------------------------------------------------------
  -- This simulates 2d6
  function Two_D return Positive is
  begin
    return One_D + One_D;
  end Two_D;

  --------------------------------------------------------------------------------------------
  -- This simulates a d4
  function D_4 return D_6_Type is
    Result : D_6_Type;
  begin
    Result := One_D;
    while Result > 4 loop
      Result := One_D;
    end loop;
    return Result;
  end D_4;

  --------------------------------------------------------------------------------------------
  -- This simulates 3d6
  function Three_D return Positive is
  begin
    return One_D + One_D + One_D;
  end Three_D;

  --------------------------------------------------------------------------------------------

  function C (Left : Ada.Strings.Unbounded.Unbounded_String; Right : String)
    return Ada.Strings.Unbounded.Unbounded_String is
  begin
    return Ada.Strings.Unbounded.To_Unbounded_String (
           Ada.Strings.Unbounded.To_String (Left) & Right);
  end C;

  --------------------------------------------------------------------------------------------
  
  function C (Left : String; Right : Ada.Strings.Unbounded.Unbounded_String)
    return Ada.Strings.Unbounded.Unbounded_String is
  begin
    return Ada.Strings.Unbounded.To_Unbounded_String (
           Left & Ada.Strings.Unbounded.To_String (Right));
  end C;

  --------------------------------------------------------------------------------------------
  
  function C (Left, Right : Ada.Strings.Unbounded.Unbounded_String)
    return Ada.Strings.Unbounded.Unbounded_String is
  begin
    return Ada.Strings.Unbounded.To_Unbounded_String (
           Ada.Strings.Unbounded.To_String (Left) &
           Ada.Strings.Unbounded.To_String (Right)
                                                     );
  end C;

  --------------------------------------------------------------------------------------------

  function Trim (The_String : in String) return String is
  begin
    return Ada.Strings.Fixed.Trim (Source => The_String, Side => Ada.Strings.Both);
  end Trim;

  --------------------------------------------------------------------------------------------
  
  procedure Select_Diet (The_Diet : in out Diet_Type) is
  begin
    -- Determine a diet for the creature
    Total := Three_D;
    If Creature.Brachiator or Creature.Tunneller or Creature.Bog_Dweller then
      Total := Total + 1;
    elsif Creature.Amphibious or Creature.Swimmer then
      Total := Total - 1;
    end if;

    --------------------------------------------------------------------------------------------
    if Creature.Environment = Desert or Creature.Secondary_Env = Desert then
        if Total in 2 .. 9 then
          The_Diet := Carnivore;
        elsif Total in 10 .. 13 then
          The_Diet := Omnivore;
        elsif Total in 14 .. 15 then
          The_Diet := Herbivore;
        else -- Total >= 16
          The_Diet := Ergivore;
        end if;
        
    elsif Creature.Environment = Tundra_Steppe or Creature.Secondary_Env = Tundra_Steppe then
        if Total in 2 .. 10 then
          The_Diet := Carnivore;
        elsif Total = 11 then
          The_Diet := Omnivore;
        elsif Total in 12 .. 16 then
          The_Diet := Herbivore;
        else -- Total >= 17
          The_Diet := Ergivore;
        end if;
        
    elsif Creature.Environment = Forest or Creature.Secondary_Env = Forest then
        if Total in 2 .. 8 then
          The_Diet := Carnivore;
        elsif Total in 9 .. 11 then
          The_Diet := Omnivore;
        elsif Total in 12 .. 16 then
          The_Diet := Herbivore;
        else -- Total >= 17
          The_Diet := Ergivore;
        end if;
        
    elsif Creature.Environment = Grassland or Creature.Secondary_Env = Grassland then
        if Total in 2 .. 9 then
          The_Diet := Carnivore;
        elsif Total = 10 then
          The_Diet := Omnivore;
        elsif Total in 11 .. 15 then
          The_Diet := Herbivore;
        else -- Total >= 16
          The_Diet := Ergivore;
        end if;
        
    elsif Creature.Environment = Unusual then
      if Creature.Unusual_Env = Aquatic then
        if Total in 2 .. 12 then
          The_Diet := Carnivore;
        elsif Total in 13 .. 14 then
          The_Diet := Omnivore;
        elsif Total in 15 .. 16 then
          The_Diet := Herbivore;
        else -- Total >= 17
          The_Diet := Ergivore;
        end if;
        
      elsif Creature.Unusual_Env = Ice_Desert then
        if Total in 2 .. 11 then
          The_Diet := Carnivore;
        elsif Total = 12 then
          The_Diet := Omnivore;
        elsif Total in 13 .. 17 then
          The_Diet := Herbivore;
        else -- Total >= 18
          The_Diet := Ergivore;
        end if;
        
      end if;
    end if;
    
  end Select_Diet;

  --------------------------------------------------------------------------------------------
  procedure Get_Carnivore_Feeding_Method (Method : in out Feeding_Type) is
  begin
  --------------------------------------------------------------------------------------------
  
    Total := Two_D;
    if Creature.Unusual_Env = Ice_Desert then
      Total := Total + 2;
    elsif Creature.Environment = Tundra_Steppe or Creature.Secondary_Env = Tundra_Steppe or
          Creature.Environment = Grassland     or Creature.Secondary_Env = Grassland     then
      Total := Total + 1;
    elsif Creature.Environment = Forest then
      Total := Total - 2;
    end if;
    if Creature.Swimmer then
      Total := Total + 1;
    end if;
    
  --------------------------------------------------------------------------------------------
  
    if Total <= 4 then
      Method := Stalker;
      Creature.Imagination    := Creature.Imagination   + 1;
      Creature.Concentration  := Creature.Concentration + 1;
      Creature.Num_Other      := Creature.Num_Other     + 1;
      Creature.Other (Creature.Num_Other) :=
        Ada.Strings.Unbounded.To_Unbounded_String ("Animal Empathy");
      Total := Two_D;
      if Total in 3 .. 10 then
        Creature.Empathy  := Creature.Empathy + 1;
      end if;
      Total := Two_D;
      if Total in 3 .. 10 then
        Creature.Num_Other := Creature.Num_Other + 1;
        Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Tracking Skill");
      end if;
      Creature.Num_Other := Creature.Num_Other + 1;
      Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Enhanced Move: 1/2 Level");

  --------------------------------------------------------------------------------------------
    elsif Total in 5 .. 7 then
      Method := Chaser;
      Creature.Num_Other := Creature.Num_Other + 1;
      Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Enhanced Move: 1 Level");
      Total := Two_D;

      Total := Two_D;
      if Total in 3 .. 8 then
        Creature.Num_Other := Creature.Num_Other + 1;
        Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Tracking Skill");
      end if;
      Total := Two_D;
      if Total in 3 .. 6 then
        Creature.Num_Other      := Creature.Num_Other     + 1;
        Creature.Other (Creature.Num_Other) :=
          Ada.Strings.Unbounded.To_Unbounded_String ("Animal Empathy");
      end if;
      Total := Two_D;
      if Total in 3 .. 6 then
        Creature.Empathy  := Creature.Empathy + 1;
      end if;
      Total := Two_D;
      if Total in 3 .. 8 then
        Creature.Gregariousness  := Creature.Gregariousness + 1;
      end if;
      
  --------------------------------------------------------------------------------------------
    elsif Total in 8 .. 9 then
      Method := Scavenger;
      
      Creature.Num_Other := Creature.Num_Other + 1;
      Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Cast Iron Stomach");
      
      Total := Two_D;
      if Total in 6 .. 9 then
        Creature.Empathy  := Creature.Empathy - 1;
      end if;
      
      Total := Two_D;
      if Total in 3 .. 6 then
        Creature.Suspicion := Creature.Suspicion + 1;
      end if;
      
  --------------------------------------------------------------------------------------------
    elsif Total in 10 .. 11 then
      Total := Two_D;
      if Total < 12 then
        Method := Pouncer;
        Total  := Two_D;
        if Total in 3 .. 10 then
          Creature.Num_Other := Creature.Num_Other + 1;
          Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Tracking Skill");
        end if;
        Creature.Concentration  := Creature.Concentration + 1;
      else
        Method := Trapper;
        Creature.Num_Other := Creature.Num_Other + 1;
        Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Alertness +1");
        Total := Two_D;
        if Total in 3 .. 10 then
          Total := One_D;
          if Total <= 3 then
            Creature.St := Creature.St + 2;
          else
            Creature.Num_Other := Creature.Num_Other + 1;
            Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Venom");
          end if;
        end if;
        
      end if;
      
  --------------------------------------------------------------------------------------------
    else -- Total >= 12
      Method := Herder;

      Total := Two_D;
      if Total in 6 .. 9 then
        Creature.Num_Other      := Creature.Num_Other     + 1;
        Creature.Other (Creature.Num_Other) :=
          Ada.Strings.Unbounded.To_Unbounded_String ("Animal Empathy");
      end if;
      
      Total := Two_D;      
      if Total in 3 .. 6 then
        Creature.Gregariousness  := Creature.Gregariousness + 2;
      end if;
      
      Total := Two_D;
      if Total in 7 .. 11 then
        Creature.Gregariousness  := Creature.Gregariousness + 1;
      end if;
      
      Total := Two_D;
      if Total in 3 .. 8 then
        Creature.Num_Other      := Creature.Num_Other     + 1;
        Creature.Other (Creature.Num_Other) :=
          Ada.Strings.Unbounded.To_Unbounded_String ("Tracking Skill");
      end if;
      
      Total := Two_D;
      if Total in 3 .. 6 then
        Creature.Num_Other      := Creature.Num_Other     + 1;
        Creature.Other (Creature.Num_Other) :=
          Ada.Strings.Unbounded.To_Unbounded_String ("Tactics Skill Bonus +1");
      end if;
      
    end if;

  --------------------------------------------------------------------------------------------
  
  end Get_Carnivore_Feeding_Method;
  
  --------------------------------------------------------------------------------------------
  
  procedure Get_Omnivore_Feeding_Method (Method : in out Feeding_Type) is
  begin
  
  --------------------------------------------------------------------------------------------
  
    Total := One_D;
    if Creature.Environment   = Forest  or
       Creature.Secondary_Env = Forest  or
       Creature.Unusual_Env   = Aquatic then
      Total := Total + 1;
    end if;

  --------------------------------------------------------------------------------------------
  
    if Total <= 2 then
      Method := Gatherer_Hunter;
      Total := Two_D;
      if Total in 6 .. 7 then
        Creature.Imagination := Creature.Imagination + 1;
      end if;
      
    elsif Total in 3 .. 4 then
      Method := Opportunist_Browser;
      Total := Two_D;
      if Total in 6 .. 9 then
        Creature.Imagination := Creature.Imagination + 1;
      end if;
      
      Total := Two_D;
      if Total in 6 .. 9 then
        Creature.Concentration := Creature.Concentration + 1;
      end if;
      
      Total := Two_D;
      if Total in 3 .. 10 then
        Creature.Num_Other := Creature.Num_Other + 1;
        Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Cast Iron Stomach");
      end if;
      
    elsif Total >= 5 then
      Method := Hunter_Browser;
      Total := Two_D;
      if Total in 3 .. 8 then
        Creature.Curiosity := Creature.Curiosity + 1;
      end if;
      
      Total := Two_D;
      if Total in 3 .. 6 then
        Creature.Imagination := Creature.Imagination + 1;
      end if;
      
    end if;
    
  --------------------------------------------------------------------------------------------
  end Get_Omnivore_Feeding_Method;
  
  --------------------------------------------------------------------------------------------
  
  procedure Get_Herbivore_Feeding_Method (Method : in out Feeding_Type) is
  begin
  
  --------------------------------------------------------------------------------------------
  
    Total := Two_D;
    if Creature.Environment = Tundra_Steppe or Creature.Secondary_Env = Tundra_Steppe or
       Creature.Environment = Grassland     or Creature.Secondary_Env = Grassland     or
       Creature.Unusual_Env = Ice_Desert    then
      Total := Total - 2;
    elsif Creature.Environment = Forest or Creature.Secondary_Env = Forest or
          Creature.Swimmer              or Creature.Fully_Aquatic          then
      Total := Total + 1;
    end if;

  --------------------------------------------------------------------------------------------

    if Total <= 4 then
      Method := Grazer;
      
      Total := Two_D;
      if Total in 3 .. 9 then
        Creature.Curiosity   := Creature.Curiosity - 1;
      end if;
      
      Creature.Imagination := Creature.Imagination - 1;
      
      Total := Two_D;
      if Total in 3 .. 6 then
        Creature.Suspicion := Creature.Suspicion + 2;
      end if;
      Total := Two_D;
      if Total in 7 .. 11 then
        Creature.Suspicion := Creature.Suspicion + 1;
      end if;
      
      Total := Two_D;
      if Total in 6 .. 9 then
        Creature.Concentration := Creature.Concentration - 1;
      end if;
      Total := Two_D;
      if Total in 3 .. 10 then
        Creature.Empathy := Creature.Empathy - 1;
      end if;
      
      Creature.Num_Other      := Creature.Num_Other + 1;
      Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Slow Eater");
      
  --------------------------------------------------------------------------------------------

    elsif Total in 5 .. 7 then
      Method := Browser;
      
      Total := Two_D;
      if Total in 6 .. 9 then
        Creature.Imagination := Creature.Imagination - 1;
      end if;
      
      Total := Two_D;
      if Total in 3 .. 6 then
        Creature.Suspicion := Creature.Suspicion + 1;
      end if;
      
      Total := Two_D;
      if Total in 6 .. 9 then
        Creature.Concentration := Creature.Concentration - 1;
      end if;
      
      Total := Two_D;
      if Total in 3 .. 10 then
        Creature.Empathy := Creature.Empathy - 1;
      end if;
      
  --------------------------------------------------------------------------------------------

    elsif Total = 8 then
      Method := Saprophytic;
      Creature.Curiosity      := Creature.Curiosity     - 2;
      Creature.Imagination    := Creature.Imagination   - 1;
      Creature.Concentration  := Creature.Concentration - 1;
      Creature.Empathy        := Creature.Empathy       - 1;
      Creature.Num_Other      := Creature.Num_Other + 1;
      Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Slow Eater");
      Creature.Num_Other      := Creature.Num_Other + 1;
      Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Cast Iron Stomach");

  --------------------------------------------------------------------------------------------

    else -- Total >= 9
      Method := Gatherer;
      
      Total := Two_D;
      if Total in 6 .. 7 then
        Creature.Imagination := Creature.Imagination - 1;
      end if;
      
      Total := Two_D;
      if Total in 6 .. 7 then
        Creature.Concentration := Creature.Concentration - 1;
      end if;
      
      Total := Two_D;
      if Total in 6 .. 9 then
        Creature.Empathy := Creature.Empathy - 1;
      end if;
      
    end if;
    
  --------------------------------------------------------------------------------------------
  
  end Get_Herbivore_Feeding_Method;
  
  --------------------------------------------------------------------------------------------

  procedure Generate_Poisons (Value  : in     Natural;
                              Poison : in out Ada.Strings.Unbounded.Unbounded_String) is
  begin
    if Value <= 1 then
      Poison := Ada.Strings.Unbounded.Null_Unbounded_String;
      
    elsif Value in 2 .. 3 then
      if One_D = 1 then
        Poison := Ada.Strings.Unbounded.To_Unbounded_String ("Poison barbs (skin agent)");
      else
        Poison := Ada.Strings.Unbounded.To_Unbounded_String ("Poison barbs");
      end if;
      
    elsif Value = 4 then
      Poison := Ada.Strings.Unbounded.To_Unbounded_String ("Irritant stench spray (mist cloud)");
      
    elsif Value = 5 then
      if One_D <= 2 then
        Poison := Ada.Strings.Unbounded.To_Unbounded_String ("Poison sweat (skin agent)");
      else
        Poison := Ada.Strings.Unbounded.To_Unbounded_String ("Poison sweat");
      end if;
      
    elsif Value in 6 .. 13 then
      Poison := Ada.Strings.Unbounded.Null_Unbounded_String;
      
    elsif Value in 14 .. 15 then
      if One_D <= 5 then
        Poison := Ada.Strings.Unbounded.To_Unbounded_String ("Poison fangs");
      else
        Poison := Ada.Strings.Unbounded.To_Unbounded_String ("Corrosive fangs");
      end if;
      
    elsif Value = 16 then
      Poison := Ada.Strings.Unbounded.To_Unbounded_String ("Stinger");
      
    elsif Value = 17 then
      if One_D <= 2 then
        Poison := Ada.Strings.Unbounded.To_Unbounded_String ("Poison spit or squirt (skin agent)");
      else
        Poison := Ada.Strings.Unbounded.To_Unbounded_String ("Poison spit or squirt");
      end if;
      
    else -- Value >= 18
      Poison := Ada.Strings.Unbounded.To_Unbounded_String ("Corrosive spit or squirt");
      
    end if;

    Creature.Poison_Potency := Creature.Ht / 4;
    
    if Creature.Swimmer or Creature.Fully_Aquatic then
      if Value = 4 or Value >= 17 then
        Poison                  := Ada.Strings.Unbounded.To_Unbounded_String ("Poison fangs");
        Creature.Poison_Potency := Creature.Poison_Potency + 2;
      end if;
    end if;

  end Generate_Poisons;

  --------------------------------------------------------------------------------------------

  -- Get the more unusual forms of creature integument. Note that the use of Other_Covering
  -- here is a bit of a cheat as the rulebook gets rather recursive in terms of what happens
  -- in the 17+ case. Given how rarely this should come up, it shouldn't be a problem...
  function Special_Integument return Ada.Strings.Unbounded.Unbounded_String is
    Temp : Natural;

    function Other_Covering return Ada.Strings.Unbounded.Unbounded_String is
      Temp2 : Natural;
    begin
      Temp2 := One_D;
      if Temp2 = 1 then
        return Ada.Strings.Unbounded.To_Unbounded_String ("Fur");
        
      elsif Temp2 = 2 then
        return Ada.Strings.Unbounded.To_Unbounded_String ("Feathers");
        
      elsif Temp2 = 3 then
        return Ada.Strings.Unbounded.To_Unbounded_String ("Skin");
        
      elsif Temp2 = 4 then
        return Ada.Strings.Unbounded.To_Unbounded_String ("Scales");
        
      elsif Temp2 = 5 then
        return Ada.Strings.Unbounded.To_Unbounded_String ("Thick Fur");
        
      else -- Temp = 6
        Temp2 := One_D;
        if Temp2 <= 3 then
          return Ada.Strings.Unbounded.To_Unbounded_String ("Thick Hide");
        else
          return Ada.Strings.Unbounded.To_Unbounded_String ("Heavy Scales");
        end if;
        
      end if;
    end Other_Covering;

  begin
    Temp := Three_D;
    
    if Temp <= 8 then
      return Ada.Strings.Unbounded.To_Unbounded_String ("Feathers Appropriate for Habitat");
      
    elsif Temp = 9 then
      return Ada.Strings.Unbounded.To_Unbounded_String ("Spiny Fur or Quills");
      
    elsif Temp = 10 then
      return Ada.Strings.Unbounded.To_Unbounded_String ("Hard Carapace (insect chitin or crustacean shell)");
      
    elsif Temp = 11 then
      return Ada.Strings.Unbounded.To_Unbounded_String ("Armour Plates (bone, very heavy chitin and so on");
      
    elsif Temp = 12 then
      return Ada.Strings.Unbounded.To_Unbounded_String ("Thick Hide");
      
    elsif Temp = 13 then
      return Ada.Strings.Unbounded.To_Unbounded_String ("Soft Carapace (insect chitin or crustacean shell)");
      
    elsif Temp = 14 then
      return Ada.Strings.Unbounded.To_Unbounded_String ("Cork or 'Barky' covering");
      
    elsif Temp = 15 then
      return Ada.Strings.Unbounded.To_Unbounded_String ("Coat of Slime");
      
    elsif Temp = 16 then
      return C (Other_Covering, " with Chameleon Properties");
      
    else -- Temp >= 17
      return C (C ("Combined ", C (Other_Covering, " and ")), Other_Covering);
      
    end if;
    
  end Special_Integument;

  --------------------------------------------------------------------------------------------

  function Voice_Box_Total return Natural is
    Temp : Natural;
  begin
    Temp := One_D;
    if Creature.Fully_Aquatic then
      Temp := Temp + 1;
    end if;
    if Creature.Society = Pack_Troop or Creature.Society = Herd or Creature.Society = Hive then
      Temp := Temp + 1;
    end if;
    if Creature.Dominant_Sex then
      Temp := Temp + 1;
    end if;
    if Creature.Feeding_Method = Herder or Creature.Feeding_Method = Gatherer_Hunter then
      Temp := Temp + 1;
    end if;
    
    return Temp;
    
  end Voice_Box_Total;

  --------------------------------------------------------------------------------------------

  function Modified_Attribute (Attribute : in Integer) return Integer is
    Temp : Integer;
  begin
    Temp := Two_D;
    if Temp in 2 .. 3 then
      return Attribute - 2;
      
    elsif Temp in 4 .. 5 then
      return Attribute - 1;
      
    elsif Temp in 6 .. 8 then
      return Attribute;
      
    elsif Temp in 9 .. 10 then
      return Attribute + 1;
      
    else -- Temp >= 11
      return Attribute + 2;
      
    end if;
    
  end Modified_Attribute;

  --------------------------------------------------------------------------------------------

  function Determine_Egotism (Egotism    : in Integer;
                              Empathy    : in Integer; 
                              Chauvinism : in Integer; 
                              Suspicion  : in Integer) return Ada.Strings.Unbounded.Unbounded_String is
  begin
    if Egotism >= 3 then
      return Ada.Strings.Unbounded.To_Unbounded_String ("Self-Centred");
      
    elsif Egotism = 2 then
      if Suspicion >= 1 or Empathy <= -1 then
        return Ada.Strings.Unbounded.To_Unbounded_String ("Self-Centred");
      else
        return Ada.Strings.Unbounded.To_Unbounded_String ("Selfish");
      end if;
      
    elsif Egotism = 1 then
      if Suspicion = 1 then
        return Ada.Strings.Unbounded.To_Unbounded_String ("Selfish");
      elsif Suspicion >= 2 or Empathy <= -2 then
        return Ada.Strings.Unbounded.To_Unbounded_String ("Self-Centred");
      else
        return Ada.Strings.Unbounded.To_Unbounded_String ("Proud (Quirk)");
      end if;
      
    elsif Egotism = 0 then
      return Ada.Strings.Unbounded.Null_Unbounded_String;
      
    elsif Egotism = -1 then
      return Ada.Strings.Unbounded.To_Unbounded_String ("Humble (Quirk)");
      
    elsif Egotism = -2 then
      if Chauvinism >= 2 then
        return Ada.Strings.Unbounded.To_Unbounded_String ("Hive Mentality");
      else
        return Ada.Strings.Unbounded.To_Unbounded_String ("Selfless");
      end if;
      
    else -- Egotism <= -3
      return Ada.Strings.Unbounded.To_Unbounded_String ("Hive Mentality");
      
    end if;
  end Determine_Egotism;
  
  --------------------------------------------------------------------------------------------

  function Get_String (The_String : Ada.Strings.Unbounded.Unbounded_String)
    return Ada.Strings.Unbounded.Unbounded_String is
    Done     : Boolean                                := False;
    Got_Char : Character                              := Ada.Characters.Latin_1.Nul;
    The_Str  : Ada.Strings.Unbounded.Unbounded_String := Ada.Strings.Unbounded.Null_Unbounded_String;
    Out_Str  : Ada.Strings.Unbounded.Unbounded_String := The_String;
    
    use type Ada.Strings.Unbounded.Unbounded_String;
    
  begin

    while not Done loop -- loop until user signals they are done
    
      Ada.Text_Io.Get_Immediate (Got_Char); -- get a character from the keyboard
      
      -- Now handle the character which has just been input
      if (Got_Char = Ada.Characters.Latin_1.Cr) or   -- i.e. a carriage return is end-of-number
         (Got_Char = Ada.Characters.Latin_1.Lf) then -- i.e. a line feed is also end-of-number
        Done := True;
        
      elsif (Got_Char = Ada.Characters.Latin_1.Bs)  or   -- accept a backspace as a delete character
            (Got_Char = Ada.Characters.Latin_1.Del) then -- or a delete as a delete character
            
        -- If the string is not null, remove the right-most character from it as the delete action
        if Ada.Strings.Unbounded.Length (The_Str) > 0 then
          The_Str := Ada.Strings.Unbounded.Head (The_Str, Ada.Strings.Unbounded.Length (The_Str) - 1);
        end if;
        
      elsif Ada.Characters.Handling.Is_Graphic (Got_Char) then -- otherwise accept non-control characters
      
        -- Add the new character onto the end of the string
        The_Str := Ada.Strings.Unbounded.To_Unbounded_String (
                   Ada.Strings.Unbounded.To_String (The_Str) & Got_Char);
                   
   -- else <ignore every other kind of character>
   
      end if;
      
      -- Display the input character
      Ada.Text_Io.Put (Got_Char);
      
    end loop;
    
    -- Output the string if string is not null
    -- (when original default string is returned unchanged)
    if The_Str /= Ada.Strings.Unbounded.Null_Unbounded_String then
      Out_Str := The_Str;
    end if;
    
    return Out_Str;
    
  end Get_String;

  --------------------------------------------------------------------------------------------

  procedure Put_Col (Prefix : in String;
                     Suffix : in String) is
  begin
    Ada.Text_Io.Put      (File => Output_File, Item => Prefix);
    Ada.Text_Io.Set_Col  (File => Output_File, To   => 45);
    Ada.Text_Io.Put      (File => Output_File, Item => ": ");
    Ada.Text_Io.Put_Line (File => Output_File, Item => Suffix);
  end Put_Col;

  --------------------------------------------------------------------------------------------

  procedure Put_Col (Prefix : in String;
                     Suffix : in Ada.Strings.Unbounded.Unbounded_String) is
  begin
    Put_Col (Prefix, Ada.Strings.Unbounded.To_String (Suffix));
  end Put_Col;

  --------------------------------------------------------------------------------------------

  procedure Divider is
  begin
    Ada.Text_Io.Put_Line (File => Output_File,
                          Item => "--------------------------------------------------------------------------------------------");
  end Divider;

  --------------------------------------------------------------------------------------------

  procedure Put_Trait (Name  : in String;
                       Trait : in Ada.Strings.Unbounded.Unbounded_String;
                       Value : in Integer) is
  begin
    if Trait = Ada.Strings.Unbounded.Null_Unbounded_String then
      Put_Col (Name, C (Trait,
                         Name & " " &
                         Trim (Integer'Image (Value))
                         ));
    else
      Put_Col (Name, C (Trait,
                         ", " & Name & " " &
                         Trim (Integer'Image (Value))
                         ));
    end if;
  end Put_Trait;

  --------------------------------------------------------------------------------------------
  
begin

  --------------------------------------------------------------------------------------------
  -- First, initialise the random number generator
  D_6.Reset (Gen => Dice);

  --------------------------------------------------------------------------------------------
  -- Determine a home environment for the creature
  Total := Two_D;
  if Total in 2 .. 3 then
    Creature.Environment := Desert;
    Creature.Num_Other   := Creature.Num_Other + 1;
    Creature.Other (Creature.Num_Other) :=
      Ada.Strings.Unbounded.To_Unbounded_String ("Survival (Desert) +1");
  elsif Total in 4 .. 5 then
    Creature.Environment := Tundra_Steppe;
  elsif Total in 6 .. 7 then
    Creature.Environment := Grassland;
  elsif Total in 8 .. 9 then
    Creature.Environment := Forest;
    Total := Three_D;
    if Total in 3 .. 6 then
      Creature.Brachiator := True;
    end if;
  elsif Total in 10 .. 11 then
    Creature.Environment := Seashore;
    Total := One_D + 1;
    If Total in 2 .. 3 then
      Creature.Secondary_Env := Desert;
    elsif Total in 4 .. 5 then
      Creature.Secondary_Env := Tundra_Steppe;
    else -- Total in 6 .. 7
      Creature.Secondary_Env := Grassland;
    end if;
    Total := Three_D;
    if Total in 3 .. 8 then
      Creature.Swimmer    := True;
    elsif Total in 9 .. 10 then
      Creature.Amphibious := True;
    end if;
  else
    -- 12 indicates an unusual environment
    Creature.Environment := Unusual;
    Total := One_D;
    if Total = 1 then
      Creature.Unusual_Env := Ice_Desert;
      Total := Three_D;
      if Total in 3 .. 10 then
        Creature.Swimmer := True;
      end if;
    elsif Total = 2 then
      Creature.Unusual_Env := Mountains_Cliff;
      Total := Three_D;
      if Total in 3 .. 10 then
        Creature.Brachiator    := True;
      elsif Total = 11 then
        Creature.Glider        := True;
      elsif Total = 12 then
        Creature.Winged_Flight := True;
      end if;
      Total := Two_D;
      if Total in 2 .. 3 then
        Creature.Secondary_Env := Desert;
      elsif Total in 4 .. 5 then
        Creature.Secondary_Env := Tundra_Steppe;
      elsif Total in 6 .. 7 then
        Creature.Secondary_Env := Grassland;
      elsif Total in 8 .. 9 then
        Creature.Secondary_Env := Forest;
      else
        Creature.Secondary_Env := Seashore;
      end if;
    elsif Total = 3 then
      Total := Three_D;
      if Total in 3 .. 10 then
        Creature.Swimmer := True;
      end if;
      Creature.Unusual_Env := Burrower;
      Creature.Tunneller   := True;
      Total := One_D + 1;
      If Total in 2 .. 3 then
        Creature.Secondary_Env := Desert;
      elsif Total in 4 .. 5 then
        Creature.Secondary_Env := Tundra_Steppe;
      else -- Total in 6 .. 7
        Creature.Secondary_Env := Grassland;
      end if;
    elsif Total = 4 then
      Creature.Unusual_Env := Swamp_Bog;
      Total := Three_D;
      if Total in 3 .. 8 then
        Creature.Swimmer    := True;
      elsif Total in 9 .. 10 then
        Creature.Amphibious := True;
      end if;
      Total := One_D;
      if Total <= 3 then
        Creature.Secondary_Env := Forest;
      else
        Creature.Secondary_Env := Grassland;
      end if;
      Creature.Bog_Dweller := True;
    elsif Total = 5 then
      Creature.Unusual_Env   := Aquatic;
      Creature.Fully_Aquatic := True;
    else
      Creature.Unusual_Env := Exotic;
    end if;
  end if;

  --------------------------------------------------------------------------------------------
  
  Select_Diet (Creature.Diet);
  
  --------------------------------------------------------------------------------------------
  -- Now get feeding type
  if Creature.Diet = Carnivore then
    Get_Carnivore_Feeding_Method (Creature.Feeding_Method);
  
  elsif Creature.Diet = Omnivore then
    Get_Omnivore_Feeding_Method (Creature.Feeding_Method);
  
  elsif Creature.Diet = Herbivore then
    Get_Herbivore_Feeding_Method (Creature.Feeding_Method);
    
  else -- must be an Ergivore
  --------------------------------------------------------------------------------------------
  
    Total := Two_D;
    if Creature.Environment = Tundra_Steppe or Creature.Secondary_Env = Tundra_Steppe or
       Creature.Unusual_Env = Ice_Desert    then
      Total := Total - 1;
    elsif Creature.Environment = Forest or Creature.Secondary_Env = Forest then
      Total := Total + 1;
    elsif Creature.Swimmer then
      Total := Total + 4;
    end if;

  --------------------------------------------------------------------------------------------
    if Total <= 4 then
      Creature.Feeding_Method := Pure;
      Creature.Imagination    := Creature.Imagination   - 1;
      Creature.Concentration  := Creature.Concentration - 1;
      Creature.Empathy        := Creature.Empathy       - 1;
      Total := Two_D;
      if Total in 6 .. 9 then
        Creature.Chauvinism := Creature.Chauvinism - 1;
      end if;

      Creature.Num_Other      := Creature.Num_Other + 1;
      Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Slow Eater");
      Creature.Num_Other      := Creature.Num_Other + 1;
      Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Doesn't Eat or Drink");
      
  --------------------------------------------------------------------------------------------

    elsif Total in 5 .. 6 then
      Creature.Feeding_Method := Mixed;
      while Creature.Secondary_Diet /= Ergivore loop
        Select_Diet (Creature.Secondary_Diet);
      end loop;
  
      --------------------------------------------------------------------------------------------
      -- Now get feeding type
      if Creature.Secondary_Diet = Carnivore then
        Get_Carnivore_Feeding_Method (Creature.Sec_Feed_Meth);
      
      elsif Creature.Secondary_Diet = Omnivore then
        Get_Omnivore_Feeding_Method (Creature.Sec_Feed_Meth);
      
      elsif Creature.Secondary_Diet = Herbivore then
        Get_Herbivore_Feeding_Method (Creature.Sec_Feed_Meth);
        
      end if;
        
      Total := Two_D;
      if Total in 6 .. 9 then
        Creature.Num_Other      := Creature.Num_Other     + 1;
        Creature.Other (Creature.Num_Other) :=
          Ada.Strings.Unbounded.To_Unbounded_String ("Animal Empathy");
      end if;
      
      Creature.Ht   := Creature.Ht   + 2;
      Creature.Hits := Creature.Hits + 2;
      
      Creature.Num_Other      := Creature.Num_Other     + 1;
      Creature.Other (Creature.Num_Other) :=
        Ada.Strings.Unbounded.To_Unbounded_String ("Regain Fatigue at Double Speed in Sunlight");
      Creature.Num_Other      := Creature.Num_Other     + 1;
      Creature.Other (Creature.Num_Other) :=
        Ada.Strings.Unbounded.To_Unbounded_String ("Starves at half the rate of ordinary creature");

  --------------------------------------------------------------------------------------------

    elsif Total in 7 .. 10 then
      Creature.Feeding_Method := Rooted;
      
      Creature.Num_Other      := Creature.Num_Other + 1;
      Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Reduced Move -1");
      
      Creature.Imagination    := Creature.Imagination   - 1;
      
      Total := Two_D;
      if Total in 3 .. 8 then
        Creature.Concentration := Creature.Concentration - 1;
      end if;
      
      Total := Two_D;
      if Total in 3 .. 8 then
        Creature.Empathy := Creature.Empathy - 1;
      end if;
      
      Total := Two_D;
      if Total in 6 .. 9 then
        Creature.Chauvinism := Creature.Chauvinism - 1;
      end if;
      
      Total := Two_D;
      if Total in 3 .. 10 then
        Creature.Num_Other      := Creature.Num_Other + 1;
        Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Doesn't Breathe");
      end if;
      
      Total := Two_D;
      if Total in 3 .. 10 then
        Creature.Num_Other      := Creature.Num_Other + 1;
        Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Slow Eater");
      end if;
      
  --------------------------------------------------------------------------------------------

    else -- Total >= 11
      Creature.Feeding_Method := Tapper;

      Total := Two_D;
      if Total in 6 .. 9 then
        Creature.Concentration := Creature.Concentration + 1;
      end if;
      
      Total := Two_D;
      if Total in 6 .. 9 then
        Creature.Imagination := Creature.Imagination + 1;
      end if;
      
      Total := Two_D;
      if Total in 3 .. 10 then
        Creature.Num_Other      := Creature.Num_Other + 1;
        Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Doesn't Breathe");
      end if;
      Creature.Num_Other      := Creature.Num_Other + 1;
      Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Doesn't Eat or Drink");

      Total := One_D;
      if Total <= 3 then
        Creature.Num_Other      := Creature.Num_Other + 1;
        Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Dependancy");
      else
        Creature.Num_Other      := Creature.Num_Other + 1;
        Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Increased Life Support");
      end if;
      
    end if;

  --------------------------------------------------------------------------------------------
  
  end if;

  --------------------------------------------------------------------------------------------
  -- Get the creature's metabolism  
  Total := One_D;
  if Creature.Environment = Tundra_Steppe or Creature.Secondary_Env = Tundra_Steppe or
     Creature.Unusual_Env = Ice_Desert                                              then
    Total := Total + 1;
  end if;
  if Creature.Fully_Aquatic then
    Total := Total - 1;
  elsif Creature.Environment = Desert or Creature.Secondary_Env = Desert then
    Total := Total - 1;
  end if;
  if Creature.Feeding_Method = Chaser or Creature.Feeding_Method = Pouncer or 
     Creature.Feeding_Method = Herder                                      then
    Total := Total + 1;
  end if;
  if Creature.Diet = Ergivore then
    Total := Total - 1;
  end if;

  if Total <= 1 then
    Creature.Metabolism := Cold_Blooded;
  elsif Total in 2 .. 5 then
    Creature.Metabolism := Warm_Blooded;
  else -- Total >= 6
    Creature.Metabolism := Hyperactive;
  end if;

  --------------------------------------------------------------------------------------------
  -- Get the creature's Society
  Total := Two_D;
  case Creature.Feeding_Method is
    when Stalker | Rooted =>
      Total := Total - 3;
    when Pouncer | Trapper =>
      Total := Total - 2;
    when Chaser =>
      Total := Total - 1;
    when Herder | Opportunist_Browser | Hunter_Browser =>
      Total := Total + 1;
    when Grazer =>
      Total := Total + 3;
    when Gatherer_Hunter | Browser | Gatherer =>
      Total := Total + 2;
    when Pure =>
      Total := Total - 4;
    when Mixed =>
      case Creature.Sec_Feed_Meth is
        when Stalker | Rooted =>
          Total := Total - 3;
        when Pouncer | Trapper =>
          Total := Total - 2;
        when Chaser =>
          Total := Total - 1;
        when Herder | Opportunist_Browser | Hunter_Browser =>
          Total := Total + 1;
        when Grazer =>
          Total := Total + 3;
        when Gatherer_Hunter | Browser | Gatherer =>
          Total := Total + 2;
        when Pure =>
          Total := Total - 4;
        when Others =>
          null;
      end case;
    when Others =>
      null;
  end case;
  
  --------------------------------------------------------------------------------------------
  if Total <= 1 then
    Creature.Society := Solitary;
    
    Total := Two_D;
    if Total in 3 .. 10 then
      Creature.Empathy := Creature.Empathy - 1;
    end if;
    
    Total := Two_D;
    if Total in 3 .. 10 then
      Creature.Egotism := Creature.Egotism + 1;
    end if;
    
    Creature.Gregariousness := Creature.Gregariousness - 1;
    Creature.Suspicion      := Creature.Suspicion      + 1;
    
  --------------------------------------------------------------------------------------------
  elsif Total in 2 .. 3 then
    Creature.Society := Pair_Bonding;
    
    Total := Two_D;
    if Total in 3 .. 6 then
      Creature.Empathy := Creature.Empathy - 1;
    end if;
    
    Total := Two_D;
    if Total in 6 .. 9 then
      Creature.Egotism := Creature.Egotism + 1;
    end if;
    
    Total := Two_D;
    if Total in 3 .. 10 then
      Creature.Gregariousness := Creature.Gregariousness - 1;
    end if;
    
    Total := Two_D;
    if Total in 3 .. 10 then
      Creature.Suspicion := Creature.Suspicion + 1;
    end if;
    
  --------------------------------------------------------------------------------------------
  elsif Total = 4 then
    Creature.Society := Family_Group;
    
    Total := Two_D;
    if Total in 3 .. 8 then
      Creature.Chauvinism := Creature.Chauvinism + 1;
    end if;
    
    Total := Two_D;
    if Total in 3 .. 6 then
      Creature.Suspicion := Creature.Suspicion + 1;
    end if;
    
  --------------------------------------------------------------------------------------------
  elsif Total in 5 .. 6 then
    Creature.Society := Pack_Troop;
    
    Total := Two_D;
    if Total in 3 .. 6 then
      Creature.Chauvinism := Creature.Chauvinism + 2;
    end if;
    
    Total := Two_D;
    if Total in 7 .. 11 then
      Creature.Chauvinism := Creature.Chauvinism + 1;
    end if;
    
    Total := Two_D;
    if Total in 3 .. 6 then
      Creature.Egotism := Creature.Egotism - 1;
    end if;
    
    Total := Two_D;
    if Total in 3 .. 6 then
      Creature.Gregariousness := Creature.Gregariousness + 1;
    end if;
    
  --------------------------------------------------------------------------------------------
  elsif Total in 7 .. 8 then
    Creature.Society := Herd;
    
    Total := Two_D;
    if Total in 7 .. 11 then
      Creature.Empathy := Creature.Empathy + 1;
    end if;
    
    Total := Two_D;
    if Total in 6 .. 9 then
      Creature.Egotism := Creature.Egotism - 1;
    end if;
    
    Total := Two_D;
    if Total in 6 .. 9 then
      Creature.Gregariousness := Creature.Gregariousness + 1;
    end if;
    
    Total := Two_D;
    if Total in 6 .. 9 then
      Creature.Chauvinism := Creature.Chauvinism + 1;
    end if;
    
    Total := Two_D;
    if Total in 6 .. 9 then
      Creature.Suspicion := Creature.Suspicion + 1;
    end if;
    
  --------------------------------------------------------------------------------------------
  else -- Total >= 9
    Creature.Society := Hive;
    
    Total := Two_D;
    if Total in 3 .. 8 then
      Creature.Empathy := Creature.Empathy + 1;
    end if;
    
    Creature.Egotism := Creature.Egotism - 1;
    
    Total := Two_D;
    if Total in 3 .. 10 then
      Creature.Gregariousness := Creature.Gregariousness + 1;
    end if;
    
    Total := Two_D;
    if Total in 3 .. 8 then
      Creature.Chauvinism := Creature.Chauvinism + 1;
    end if;
    
    Creature.Suspicion := Creature.Suspicion + 1;
    
  end if;

  --------------------------------------------------------------------------------------------
  -- Get the creature's Size
  Total := Two_D;
  if Creature.Unusual_Env = Ice_Desert then
    Total := Total - 1;
  end if;
  if Creature.Feeding_Method = Gatherer_Hunter or 
     Creature.Feeding_Method = Pouncer         then
    Total := Total - 1;
  end if;
  if Creature.Feeding_Method = Gatherer or
     Creature.Feeding_Method = Pure     or 
     Creature.Feeding_Method = Rooted   or 
     Creature.Feeding_Method = Tapper   then
    Total := Total - 2;
  end if;
  if Creature.Brachiator then
    Total := Total - 2;
  end if;
    
  if Creature.Unusual_Env = Burrower then
    Total := Total - 3;
  end if;
  if Creature.Feeding_Method = Grazer then
    Total := Total + 2;
  end if;
  if Creature.Fully_Aquatic then
    Total := Total + 1;
  end if;
  
  if Total <= 2 then
    Creature.Size := Tiny;
    Creature.Num_Other      := Creature.Num_Other + 1;
    Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Inconvenient Size (-15)");
    Creature.Num_Other      := Creature.Num_Other + 1;
    Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Decreased Life Support (+10)");
  elsif Total = 3 then
    Creature.Size := Small;
  elsif Total in 4 .. 5 then
    Creature.Size := Short;
  elsif Total in 6 .. 8 then
    Creature.Size := Average;
  elsif Total in 9 .. 10 then
    Creature.Size := Tall;
  elsif Total = 11 then
    Creature.Size := Big;
  else -- Total >= 12
    Creature.Size := Huge;
    Creature.Num_Other      := Creature.Num_Other + 1;
    Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Inconvenient Size (-10)");
    Creature.Num_Other      := Creature.Num_Other + 1;
    Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Increased Life Support (-20)");
    Total := Three_D;
    if Total <= 5 then
      Creature.Incr_Strength := True;
    end if;
    Creature.Suspicion := Creature.Suspicion - 1;
--  else -- Must be Giant
--    Inconvenient Size (-15)
--    Increased Life Support (-20)
--    On 3d roll of 6- they have Increased Strength
--    Suspicion -2
  end if;

  Creature.Avg_Weight := Base_Weight (Creature.Size);
  Creature.St         := Creature.St   + Size_St_Mod   (Creature.Size);
  Creature.Ht         := Creature.Ht   + Size_Ht_Mod   (Creature.Size);
  Creature.Hits       := Creature.Hits + Size_Hits_Mod (Creature.Size);
  
  --------------------------------------------------------------------------------------------
  -- Get the creature's Bulk
  Total := One_D;
  if Creature.Feeding_Method = Grazer then
    Total := Total + 1;
  end if;
  if Creature.Unusual_Env = Ice_Desert then
    Total := Total + 1;
  elsif Creature.Environment = Desert or Creature.Secondary_Env = Desert then
    Total := Total - 1;
  end if;
  if Creature.Brachiator then
    Total := Total - 1;
  end if;
  if Creature.Metabolism = Hyperactive then
    Total := Total - 1;
  end if;
  if Creature.Incr_Strength then
    Total := Total + 1;
  end if;
  
  if Total <= 1 then
    Creature.Bulk := Slight;
  elsif Total = 2 then
    Creature.Bulk := Light;
  elsif Total in 3 .. 4 then
    Creature.Bulk := Average;
  elsif Total = 5 then
    Creature.Bulk := Heavy;
  else -- Total >= 6
    Creature.Bulk := Bulky;
  end if;
  
  Creature.Avg_Weight := Integer (Float (Creature.Avg_Weight) * Bulk_Multiplier (Creature.Bulk));
  Creature.Ht         := Creature.Ht + Bulk_Ht_Mod (Creature.Bulk);

  --------------------------------------------------------------------------------------------
  -- Generate position in food chain
  Total := One_D;
  if Creature.Diet = Carnivore then
    Total := Total + 2;
  elsif Creature.Diet = Herbivore then
    Total := Total - 1;
  end if;
  if Creature.Swimmer then
    Total := Total + 1;
  end if;
  if Creature.Size = Tiny or Creature.Size = Small then
    Total := Total - 1;
  elsif Creature.Size = Big then
    Total := Total + 1;
  elsif Creature.Size = Huge or Creature.Size = Giant then
    Total := Total + 2;
  end if;
  
  --------------------------------------------------------------------------------------------
  if Total <= 1 then
    Creature.Chain_Posn                 := Low;
    Creature.Num_Other                  := Creature.Num_Other + 1;
    Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Stealth (DX)");
    Creature.Suspicion                  := Creature.Suspicion + 2;
  elsif Total in 2 .. 3 then
    Creature.Chain_Posn := Middle;
    Creature.Suspicion  := Creature.Suspicion + 1;
  elsif Total in 4 .. 5 then
    Creature.Chain_Posn := Near_Top;
    
    Total := Two_D;
    if Total in 3 .. 7 then
      Creature.Num_Other                  := Creature.Num_Other + 1;
      Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Animal Empathy");
    end if;
    
  elsif Total in 6 .. 7 then
    Creature.Chain_Posn := Top_Shared;
    
    Total := Two_D;
    if Total in 3 .. 9 then
      Creature.Chauvinism := Creature.Chauvinism + 1;
    end if;
    
    Total := Two_D;
    if Total in 3 .. 9 then
      Creature.Empathy    := Creature.Empathy    + 1;
    end if;
    
  elsif Total >= 8 then
    Creature.Chain_Posn := Top_Solitary;
    Creature.Suspicion  := Creature.Suspicion - 1;
    Creature.Empathy    := Creature.Empathy   - 1;
  end if;

  --------------------------------------------------------------------------------------------
  -- Generate reproductive strategy
  Total := One_D;
  if Creature.Size = Tiny or Creature.Size = Small then
    Total := Total - 1;
  elsif Creature.Size = Huge or Creature.Size = Giant then
    Total := Total + 1;
  end if;
  if Creature.Chain_Posn = Low then
    Total := Total - 1;
  elsif Creature.Chain_Posn = Top_Shared or Creature.Chain_Posn = Top_Solitary then
    Total := Total + 1;
  end if; 

  if Total <= 1 then
    Creature.Offspring   := Hordes_Very_Little;
    Creature.Egotism     := Creature.Egotism     - 2;
    Creature.Curiosity   := Creature.Curiosity   - 2;
    Creature.Imagination := Creature.Imagination - 1;
  elsif Total in 2 .. 3 then
    Creature.Offspring   := Many_Low;
    Creature.Egotism     := Creature.Egotism     - 1;
    Creature.Imagination := Creature.Imagination - 1;
    Creature.Curiosity   := Creature.Curiosity   - 1;
  elsif Total in 4 .. 5 then
    Creature.Offspring := Several_Moderate;
  elsif Total = 6 then
    Creature.Offspring := Few_Much;
    Total := Two_D;
    if Total in 3 .. 6 then
      Creature.Egotism     := Creature.Egotism     + 1;
    end if;
    Total := Two_D;
    if Total in 5 .. 9 then
      Creature.Empathy     := Creature.Empathy     + 1;
    end if;
    Total := Two_D;
    if Total in 3 .. 10 then
      Creature.Curiosity   := Creature.Curiosity   + 1;
    end if;
    Total := Two_D;
    if Total in 3 .. 6 then
      Creature.Imagination := Creature.Imagination + 1;
    end if;
    
  else -- Total >= 7
    Creature.Offspring := Very_Few_Very_Much;
    
    Total := Two_D;
    if Total in 3 .. 10 then
      Creature.Empathy := Creature.Empathy + 1;
    end if;
    Total := Two_D;
    if Total in 6 .. 9 then
      Creature.Egotism := Creature.Egotism + 1;
    end if;
    Total := Two_D;
    if Total in 3 .. 6 then
      Creature.Concentration := Creature.Concentration + 1;
    end if;
    Creature.Curiosity := Creature.Curiosity + 1;
    Total := Two_D;
    if Total in 3 .. 8 then
      Creature.Imagination := Creature.Imagination + 1;
    end if;
    
  end if;

  --------------------------------------------------------------------------------------------
  -- Generate lifespan
  Total := Two_D;
  if Creature.Size = Huge or Creature.Size = Giant then
    Total := Total + 1;
  elsif Creature.Size = Tiny or Creature.Size = Small then
    Total := Total - 1;
  end if;
  if Creature.Offspring = Hordes_very_little then
    Total := Total - 2;
  elsif Creature.Offspring = Many_Low then
    Total := Total - 1;
  elsif Creature.Offspring = Few_Much then
    Total := Total + 1;
  elsif Creature.Offspring = Very_Few_Very_Much then
    Total := Total + 2;
  end if;
  
  if Total <= 2 then
    Creature.Lifespan := Ada.Strings.Unbounded.To_Unbounded_String ("Short Lifespan (Two Levels)");
  elsif Total in 3 .. 4 then
    Creature.Lifespan := Ada.Strings.Unbounded.To_Unbounded_String ("Short Lifespan (One Level)");
  elsif Total in 5 .. 10 then
    Creature.Lifespan := Ada.Strings.Unbounded.To_Unbounded_String ("Normal Lifespan");
  elsif Total in 11 .. 12 then
    Creature.Lifespan := Ada.Strings.Unbounded.To_Unbounded_String ("Extended Lifespan (One Level)");
  else -- Total >= 13
    Creature.Lifespan := Ada.Strings.Unbounded.To_Unbounded_String ("Extended Lifespan (Two Levels)");
  end if;
  
  --------------------------------------------------------------------------------------------
  -- Generate number of sexes
  Total := Two_D;
  if Total = 2 then
    Creature.Genders := Ada.Strings.Unbounded.To_Unbounded_String ("Two sexes; gender changes with season or age");
    Creature.Empathy := Creature.Empathy + 1;
  elsif Total in 3 .. 8 then
    Creature.Genders := Ada.Strings.Unbounded.To_Unbounded_String ("Two sexes");
  elsif Total = 9 then
    Creature.Genders := Ada.Strings.Unbounded.To_Unbounded_String ("Hermaphroditic; one sex performs both roles; mating required");
  elsif Total = 10 then
    Creature.Genders := Ada.Strings.Unbounded.To_Unbounded_String ("Hermaphroditic; self-contained");
    Total := Two_D;
    if Total in 3 .. 6 then
      Creature.Gregariousness := Creature.Gregariousness - 1;
    end if;
  elsif Total = 11 then
    Creature.Genders := Ada.Strings.Unbounded.To_Unbounded_String ("Three sexes");
    Total := Two_D;
    if Total in 3 .. 6 then
      Creature.Gregariousness := Creature.Gregariousness + 1;
    end if;
  else -- Total = 12
    Total := One_D + 3;
    Creature.Genders := Ada.Strings.Unbounded.To_Unbounded_String (Trim (Integer'Image (Total)) & 
                                                                   " sexes; sex education a college-level course; may be an artificial arrangement");
    Creature.Gregariousness := Creature.Gregariousness + 1;
    Total := Two_D;
    if Total in 3 .. 9 then
      Creature.Imagination := Creature.Imagination + 1;
    end if;
  end if;
  
  --------------------------------------------------------------------------------------------
  -- Generate method of delivery
  Total := Two_D;
  if Creature.Metabolism = Cold_Blooded then
    Total := Total + 2;
  end if;
  if Creature.Offspring in Several_Moderate .. Very_Few_Very_Much then
    Total := Total - 2;
  end if;
  
  if Total <= 4 then
    Creature.Delivery := Ada.Strings.Unbounded.To_Unbounded_String ("Live birth; born very weak and defenceless");
    Total := Two_D;
    if Total in 3 .. 10 then
      Creature.Empathy := Creature.Empathy + 1;
    end if;
    
  elsif Total = 5 then
    Creature.Delivery := Ada.Strings.Unbounded.To_Unbounded_String ("Live birth; kept in pouch for part of childhood");
    Total := Two_D;
    if Total in 6 .. 9 then
      Creature.Empathy := Creature.Empathy + 1;
    end if;
    
  elsif Total in 6 .. 7 then
    Creature.Delivery := Ada.Strings.Unbounded.To_Unbounded_String ("Live birth; born fairly well-developed");
    
  elsif Total = 8 then
    Creature.Delivery := Ada.Strings.Unbounded.To_Unbounded_String ("Inconvenient (-5 Compulsion or Dependancy)");
    
  elsif Total in 9 .. 10 then
    Creature.Delivery := Ada.Strings.Unbounded.To_Unbounded_String ("Oviparous (lays eggs)");
    
    Total := Two_D;
    if Total in 6 .. 9 then
      Creature.Empathy := Creature.Empathy + 1;
    end if;
    
  elsif Total = 11 then
    Creature.Delivery := Ada.Strings.Unbounded.To_Unbounded_String ("Buds; small creatures grow from parent");
    
  elsif Total = 12 then
    Creature.Delivery := Ada.Strings.Unbounded.To_Unbounded_String ("Fissions; creature splits into two or more young");
    Total := Two_D;
    if Total in 3 .. 10 then
      Creature.Egotism := Creature.Egotism - 1;
    end if;
    Total := Two_D;
    if Total in 11 .. 12 then
      Creature.Egotism := Creature.Egotism - 2;
    end if;
    
  else -- Total >= 13
    Creature.Delivery := Ada.Strings.Unbounded.To_Unbounded_String ("Weird or Disgusting (-10 point Compulsive Behaviour or Odious Personal Habit)");
    
  end if;

  --------------------------------------------------------------------------------------------
  -- Generate special features of species reproductive method
  for Ctr in 1 .. 2 loop
    Total := Three_D;
    
    if Total in 3 .. 5 then
      null; -- Nothing special
      
    elsif Total in 6 .. 8 then
      Creature.Reprod_Spec (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Sex plays an important part in species social system");
      Total := Two_D;
      if Total in 3 .. 10 then
        Creature.Gregariousness := Creature.Gregariousness + 1;
      end if;
      Total := Two_D;
      if Total in 7 .. 9 then
        Creature.Egotism := Creature.Egotism + 1;
      end if;
      
    elsif Total = 9 then
      exit;
      
    elsif Total = 10 then
      Creature.Reprod_Spec (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Only dominant members of society reproduce");
      Total := Two_D;
      if Total in 3 .. 10 then
        Creature.Gregariousness := Creature.Gregariousness + 1;
      end if;
      Total := Two_D;
      if Total in 3 .. 10 then
        Creature.Egotism := Creature.Egotism + 1;
      end if;
      Creature.Dominant_Sex := True;
      
    elsif Total = 11 then
      Creature.Reprod_Spec (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Only dominant members of society reproduce, and subdominants are sterile/neuters with Egotism - 1");
      Total := Two_D;
      if Total in 3 .. 10 then
        Creature.Gregariousness := Creature.Gregariousness + 1;
      end if;
      Total := Two_D;
      if Total in 3 .. 10 then
        Creature.Egotism := Creature.Egotism + 1;
      end if;
      Creature.Dominant_Sex := True;
      
    elsif Total = 12 then
      Creature.Reprod_Spec (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Only one female per social group is fertile");
      
    elsif Total = 13 then
      Creature.Reprod_Spec (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("High sexual dimorphism; males and females are physically very different");

    elsif Total in 14 .. 15 then
      if not Creature.Casted then
        Total := One_D + 1;
        Creature.Reprod_Spec (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Species has " &
                                                                                 Trim (Integer'Image (Total)) &
                                                                                 " genetically determined castes");
        if Total = 15 then
          Creature.Reprod_Spec (Ctr) := C (Creature.Reprod_Spec (Ctr), "; only one caste can reproduce; ");
          Total := One_D;
          if Total <= 3 then
            Creature.Reprod_Spec (Ctr) := C (Creature.Reprod_Spec (Ctr), "'Queens' are mindless");
          else
            Creature.Reprod_Spec (Ctr) := C (Creature.Reprod_Spec (Ctr), "'Queens' are highly individualistic; the 'Queen' has Egotism +2 while all others have Egotism -1");
            Creature.Individ_Queens    := True;
          end if;
        end if;
        
        Creature.Chauvinism := Creature.Chauvinism + 1;
        Creature.Casted     := True;
      end if;
      
    elsif Total = 16 then
      Creature.Reprod_Spec (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Offspring have a lengthy 'mindless' or 'animal' phase during development");
      Total := Two_D;
      if Total in 3 .. 7 then
        Creature.Egotism := Creature.Egotism + 1;
      end if;
      
    else -- Total >= 17
      Creature.Reprod_Spec (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Something REALLY weird, inconvenient and possibly disgusting; " &
                                                                               "may lay eggs in host creatures; mating may be fatal");

    end if;

  end loop;

  -- Handle special case
  if Creature.Reprod_Spec (1) = Ada.Strings.Unbounded.To_Unbounded_String ("High sexual dimorphism; males and females are physically very different") and
     Creature.Reprod_Spec (1) = Creature.Reprod_Spec (2) then
     
    Creature.Reprod_Spec (2) := Ada.Strings.Unbounded.Null_Unbounded_String;
    Creature.Reprod_Spec (1) := Ada.Strings.Unbounded.To_Unbounded_String ("High sexual dimorphism; males and females are physically very different; only one sex is sentient");
    
    Total := Two_D;
    if Total in 3 .. 6 then
      Creature.Empathy := Creature.Empathy - 1;
    end if;

  end if;
  
  -- Remove duplicates
  if Creature.Reprod_Spec (1) = Creature.Reprod_Spec (2) then
    Creature.Reprod_Spec (2) := Ada.Strings.Unbounded.Null_Unbounded_String;
  end if;

  --------------------------------------------------------------------------------------------
  -- Generate basic shape of creature
  Total := Two_D;
  if Total = 2 then
    Total := One_D;
    if Total <= 3 then
      Creature.Num_Limbs := 0;
    else
      Creature.Num_Limbs := 1;
    end if;
    Sym_Number := 3;
    
  elsif Total = 3 then
    Creature.Num_Limbs := 2;
    Sym_Number         := 1;
    
  elsif Total = 4 then
    Creature.Num_Limbs := 3;
    Sym_Number         := 4;
    
  elsif Total in 5 .. 8 then
    Creature.Num_Limbs := 4;
    Sym_Number         := 1;
    
  elsif Total in 9 .. 11 then
    Creature.Num_Limbs := 6;
    Sym_Number         := 2;
    
  else -- Total = 12
    Creature.Num_Limbs := One_D + 6;
    Sym_Number         := 3;
    
  end if;
  
  Total := One_D;
  if Total <= Sym_Number then
    Creature.Symmetry := Radial;
  else
    Creature.Symmetry := Bilateral;
  end if;
  
  --------------------------------------------------------------------------------------------
  -- Get any extra limbs
  Total := One_D;
  If Creature.Num_Limbs <= 1 then
    Total := Total + 1;
  end if;

  -- Get number of extra limbs
  if Total <= 2 then
    Creature.Num_Extra_Limbs := 0;
    
  elsif Total in 3 .. 4 then
    Creature.Num_Extra_Limbs := 1;
    
  elsif Total = 5 then 
    Creature.Num_Extra_Limbs := One_D + 3;
    
  else -- Total >= 6
    Creature.Num_Extra_Limbs := 50; -- Actually many, all with flimsy strength
    
  end if;

  -- Get strength of extra limbs
  if Creature.Num_Extra_Limbs = 50 then
    Creature.Ext_Limb_St := Ada.Strings.Unbounded.To_Unbounded_String ("Flimsy (as feelers); ST 1");
    
  else
    Total := One_D;
    if Total <= 2 then
      Creature.Ext_Limb_St := Ada.Strings.Unbounded.To_Unbounded_String ("Flimsy (as feelers); ST 1");
      
    elsif Total in 3 .. 4 then
      Creature.Ext_Limb_St := Ada.Strings.Unbounded.To_Unbounded_String ("Weak (as human lips); ST 2");
      
    elsif Total = 5 then
      Creature.Ext_Limb_St := Ada.Strings.Unbounded.To_Unbounded_String ("Fair (as monkey tail); ST -4");
      
    else -- Total = 6
      Creature.Ext_Limb_St := Ada.Strings.Unbounded.To_Unbounded_String ("Strong (as elephant's trunk); ST as primary limbs");
      
    end if;
  end if;
  
  -- Get dexterity of extra limbs
  Total := One_D;
  if Total <= 3 then
    Creature.Ext_Limb_Dx := Ada.Strings.Unbounded.To_Unbounded_String ("Crude; DX -4");
    
  elsif Total in 4 .. 5 then
    Creature.Ext_Limb_Dx := Ada.Strings.Unbounded.To_Unbounded_String ("Fair; DX as primary limbs");
    
  else -- Total = 6
    Creature.Ext_Limb_Dx := Ada.Strings.Unbounded.To_Unbounded_String ("Excellent; DX +2");
    
  end if;
  
  -- Get tactile sensitivity of extra limbs
  Total := One_D;
  if Total <= 2 then
    Creature.Ext_Limb_Sens := Ada.Strings.Unbounded.To_Unbounded_String ("Crude");
    
  elsif Total in 3 .. 4 then
    Creature.Ext_Limb_Sens := Ada.Strings.Unbounded.To_Unbounded_String ("Fair");
    
  elsif Total = 5 then
    Creature.Ext_Limb_Sens := Ada.Strings.Unbounded.To_Unbounded_String ("Excellent");
    
  else -- Total = 6
    Creature.Ext_Limb_Sens := Ada.Strings.Unbounded.To_Unbounded_String ("Fair; has excellent other sense (usually taste or smell) built in");
    
  end if;

  --------------------------------------------------------------------------------------------
  -- Generate posture
  Total := One_D;
  if Creature.Symmetry = Bilateral then
    if Creature.Environment   = Forest or Creature.Secondary_Env = Forest then
      Total := Total + 1;
    end if;
    if Creature.Brachiator then
      Total := Total + 1;
    end if;
    if Creature.Swimmer then
      Total := Total - 1;
    end if;
    if Creature.Amphibious or Creature.Fully_Aquatic then
      Total := Total - 2;
    end if;

    if Total <= 2 then
      Creature.Posture := Horizontal;
      
    elsif Total = 3 then
      if Creature.Num_Limbs >= 6 then
        Creature.Posture := Centauroid;
      else
        Creature.Posture := Horizontal;
      end if;
      
    elsif Total in 4 .. 5 then
      if Creature.Num_Limbs >= 8 then
        Creature.Posture := Centauroid;
      else
        Creature.Posture := Semi_Upright;
      end if;
      
    else -- Total >= 6
      Creature.Posture := Upright;
      
    end if;

  else -- must be radially symmetrical
  
    if Creature.Brachiator then
      Total := Total + 2;
    end if;
    if Creature.Swimmer or Creature.Amphibious or Creature.Fully_Aquatic then
      Total := Total - 1;
    end if;

    if Total in 0 .. 2 then
      Creature.Posture := Invertebrate;
      
    elsif Total in 3 .. 5 then
      Creature.Posture := Semi_Upright;
      
    else -- Total >= 6
      Creature.Posture := Upright;
      
    end if;
    
  end if;
  
  -- Divide the creature's limbs between walking and handling
  case Creature.Posture is
    when Horizontal =>
      Creature.Walking_Limbs  := Creature.Num_Limbs;
      Creature.Handling_Limbs := 0;
      
    when Semi_Upright =>
      if Creature.Num_Limbs >= 2 then
        Creature.Walking_Limbs  := Creature.Num_Limbs - 2;
        Creature.Handling_Limbs := 2;
      else
        Creature.Walking_Limbs  := 0;
        Creature.Handling_Limbs := Creature.Num_Limbs;
      end if;
      
    when Centauroid =>
      if Creature.Num_Limbs in 6 .. 7 then
        Creature.Walking_Limbs  := Creature.Num_Limbs - 2;
        Creature.Handling_Limbs := 2;
      else
        Total := One_D;
        if Total <= 3 then
          Creature.Walking_Limbs  := Creature.Num_Limbs - 2;
          Creature.Handling_Limbs := 2;
        else
          Creature.Walking_Limbs  := Creature.Num_Limbs - 4;
          Creature.Handling_Limbs := 4;
        end if;
      end if;
      
    when Upright =>
      if Creature.Num_Limbs >= 2 then
        Creature.Walking_Limbs  := 2;
        Creature.Handling_Limbs := Creature.Num_Limbs - 2;
      else
        Creature.Walking_Limbs  := Creature.Num_Limbs;
        Creature.Handling_Limbs := 0;
      end if;
      
    when Invertebrate =>
      Creature.Walking_Limbs  := 0;
      Creature.Handling_Limbs := Creature.Num_Limbs;
      
  end case;

  -- Get the walking limb type
  Total := One_D;
  If Creature.Swimmer then
    Total := Total - 1;
  end if;
  if Creature.Amphibious or Creature.Fully_Aquatic then
    Total := Total - 2;
  end if;
  if Creature.Brachiator then
    Total := Total + 1;
  end if;
  if Creature.Environment = Grassland     or Creature.Secondary_Env = Grassland     or
     Creature.Environment = Tundra_Steppe or Creature.Secondary_Env = Tundra_Steppe then
    Total := Total - 1;
  end if;
  
  if Total <= 2 then
    if Creature.Amphibious or Creature.Fully_Aquatic then
      Creature.Walk_Limb_Type := Ada.Strings.Unbounded.To_Unbounded_String ("Large Fluke");
    else
      Creature.Walk_Limb_Type := Ada.Strings.Unbounded.To_Unbounded_String ("Running Hooves");
    end if;
    Creature.Num_Other := Creature.Num_Other + 1;
    Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Enhanced Move +1");
    
  elsif Total = 3 then
    Creature.Walk_Limb_Type := Ada.Strings.Unbounded.To_Unbounded_String ("Cloven Hoof");
    Creature.Dx             := Creature.Dx + 1;
    
  elsif Total in 4 .. 5 then
    Creature.Walk_Limb_Type := Ada.Strings.Unbounded.To_Unbounded_String ("Walking Paws");
    
  elsif Total = 6 then
    Creature.Walk_Limb_Type := Ada.Strings.Unbounded.To_Unbounded_String ("Grasping Paws");
    Creature.Num_Other := Creature.Num_Other + 1;
    Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Reduced Move -1");
    
  else -- Total >= 7
    Creature.Walk_Limb_Type := Ada.Strings.Unbounded.To_Unbounded_String ("Crude Hands");
    Creature.Num_Other := Creature.Num_Other + 1;
    Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Reduced Move -1");
    
  end if;
  
  -- Get the handling limb type
  Total := One_D;
  if Creature.Posture = Invertebrate or Creature.Posture = Horizontal then
    Total := Total - 1;
  elsif Creature.Posture = Centauroid or Creature.Posture = Upright then
    Total := Total + 1;
  end if;
  if Creature.Brachiator then
    Total := Total + 1;
  end if;
  
  if Total <= 2 then
    Creature.Hand_Limb_Type := Ada.Strings.Unbounded.To_Unbounded_String ("Walking Paws");
    Creature.Num_Other := Creature.Num_Other + 1;
    Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Half Level Enhanced Move when walking on all limbs");
    
  elsif Total in 3 .. 5 then
    Creature.Hand_Limb_Type := Ada.Strings.Unbounded.To_Unbounded_String ("Grasping Paws");
    
  elsif Total in 6 .. 7 then
    Creature.Hand_Limb_Type := Ada.Strings.Unbounded.To_Unbounded_String ("Crude Hands");
    
  else -- Total >= 8
    Creature.Hand_Limb_Type := Ada.Strings.Unbounded.To_Unbounded_String ("Hands");
    
  end if;
  
  --------------------------------------------------------------------------------------------
  -- Get natural physical weapons
  for Ctr in 1 .. 2 loop
    Total := Two_D;
    if Creature.Fully_Aquatic then
      Total := Total + 2;
    end if;
    if Creature.Diet = Carnivore then
      Total := Total + 2;
    elsif Creature.Diet = Herbivore then
      Total := Total - 2;
    end if;
    if Creature.Size = Huge or Creature.Size = Giant then
      Total := Total - 1;
    end if;
    
    if Total <= 0 then
      Creature.Natural_Weapons (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Large Horns or Tusks");
      
    elsif Total = 1 then
      Creature.Natural_Weapons (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Spear");
      
    elsif Total = 2 then
      Creature.Natural_Weapons (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Smashing Tail");
      
    elsif Total = 3 then
      Creature.Natural_Weapons (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Butting Horns");
      
    elsif Total = 4 then
      Creature.Natural_Weapons (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Horns");
      
    elsif Total = 5 then
      Creature.Natural_Weapons (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Blunt Teeth");
      
    elsif Total in 6 .. 9 then
      null;
      
    elsif Total = 10 then
      Creature.Natural_Weapons (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Sharp Teeth");
      
    elsif Total = 11 then
      Creature.Natural_Weapons (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Small Claws");
      
    elsif Total = 12 then
      Creature.Natural_Weapons (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Fangs");
      
    elsif Total = 13 then
      Creature.Natural_Weapons (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Large Claws");
      
    elsif Total = 14 then
      Creature.Natural_Weapons (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Smashing Tail");
      
    else -- Total >= 15
      Creature.Natural_Weapons (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Spear");
      
    end if;
    
  end loop;
  
  -- Remove duplicates
  if Creature.Natural_Weapons (1) = Creature.Natural_Weapons (2) then
    Creature.Natural_Weapons (2) := Ada.Strings.Unbounded.Null_Unbounded_String;
  end if;
  
  --------------------------------------------------------------------------------------------
  -- Get natural poisons and venoms
  Total := Three_D;
  
  if Creature.Chain_Posn = Low then
    Total := Total - 3;
  elsif Creature.Chain_Posn = Middle then
    Total := Total - 1;
  end if;
  if Creature.Environment = Desert or Creature.Secondary_Env = Desert then
    Total := Total + 2;
  end if;
  if Creature.Amphibious or Creature.Swimmer or Creature.Fully_Aquatic then
    Total := Total + 1;
  end if;
  if Creature.Size = Huge or Creature.Size = Giant then
    Total := Total - 1;
  end if;

  if Total <= 1 then
    for Ctr in 1 .. 2 loop
      Generate_Poisons (One_D, Creature.Natural_Poisons (Ctr));
    end loop;
  else
    Generate_Poisons (Total, Creature.Natural_Poisons (1));
  end if;

  --------------------------------------------------------------------------------------------
  -- Get extraordinary natural weapons
  Total := Three_D;
  if Creature.Feeding_Method = Trapper or Creature.Diet = Herbivore then
    Total := Total - 1;
  elsif Creature.Diet = Ergivore then
    Total := Total + 2;
  end if;
  if Creature.Fully_Aquatic then
    Total := Total + 1;
  end if;
  
  if Total <= 3 then
    Creature.Extrord_Weapons := Ada.Strings.Unbounded.To_Unbounded_String ("Illusion");
    
  elsif Total = 4 then
    if Creature.Fully_Aquatic then
      Creature.Extrord_Weapons := Ada.Strings.Unbounded.To_Unbounded_String ("Ink");
    else
      Creature.Extrord_Weapons := Ada.Strings.Unbounded.To_Unbounded_String ("Smoke");
    end if;
    
  elsif Total in 5 .. 14 then
    null;
    
  elsif Total = 15 then
    Creature.Extrord_Weapons := Ada.Strings.Unbounded.To_Unbounded_String ("Deafen");
    
  elsif Total = 16 then
    Creature.Extrord_Weapons := Ada.Strings.Unbounded.To_Unbounded_String ("Sonic Blast");
    
  elsif Total = 17 then
    if Creature.Fully_Aquatic then
      Creature.Extrord_Weapons := Ada.Strings.Unbounded.To_Unbounded_String ("Ink");
    else
      Creature.Extrord_Weapons := Ada.Strings.Unbounded.To_Unbounded_String ("Smoke");
    end if;
    
  elsif Total = 18 then
    Creature.Extrord_Weapons := Ada.Strings.Unbounded.To_Unbounded_String ("Flash");
    
  elsif Total = 19 then
    Creature.Extrord_Weapons := Ada.Strings.Unbounded.To_Unbounded_String ("Electrical Charge");
    
  else -- Total >= 20
    Creature.Extrord_Weapons := Ada.Strings.Unbounded.To_Unbounded_String ("Lightning Missile (Laser)");
    
  end if;
  
  --------------------------------------------------------------------------------------------
  -- Generate body covering
  Total := One_D;
  
  -- For hot climates
  if Creature.Environment = Desert or Creature.Secondary_Env = Desert then
    if Total in 1 .. 2 then
      Creature.Integument := Ada.Strings.Unbounded.To_Unbounded_String ("Very Thin Fur");
      
    elsif Total in 3 .. 4 then
      Creature.Integument := Ada.Strings.Unbounded.To_Unbounded_String ("Skin");
      
    elsif Total = 5 then
      Creature.Integument := Ada.Strings.Unbounded.To_Unbounded_String ("Scales");
      
    else -- Total = 6
      Creature.Integument := Special_Integument;
      
    end if;
    
  -- For moderate climates
  elsif Creature.Environment = Grassland or Creature.Secondary_Env = Grassland or
        Creature.Environment = Forest    or Creature.Secondary_Env = Forest    then
    if Total = 1 then
      Creature.Integument := Ada.Strings.Unbounded.To_Unbounded_String ("Fur");
      
    elsif Total = 2 then
      Creature.Integument := Ada.Strings.Unbounded.To_Unbounded_String ("Feathers");
      
    elsif Total = 3 then
      Creature.Integument := Ada.Strings.Unbounded.To_Unbounded_String ("Skin");
      
    elsif Total = 4 then
      Creature.Integument := Ada.Strings.Unbounded.To_Unbounded_String ("Heavy Scales");
      
    elsif Total = 5 then
      Creature.Integument := Ada.Strings.Unbounded.To_Unbounded_String ("Thick Fur");
      
    else -- Total = 6
      Creature.Integument := Special_Integument;
      
    end if;

  -- For cold climates
  elsif Creature.Environment = Tundra_Steppe or Creature.Secondary_Env = Tundra_Steppe or
        Creature.Unusual_Env = Ice_Desert then
    if Total = 1 then
      Creature.Integument := Ada.Strings.Unbounded.To_Unbounded_String ("Fur");
      
    elsif Total = 2 then
      Creature.Integument := Ada.Strings.Unbounded.To_Unbounded_String ("Skin");
      
    elsif Total = 3 then
      Creature.Integument := Ada.Strings.Unbounded.To_Unbounded_String ("Thick Hide");
      
    elsif Total in 4  .. 5 then
      Creature.Integument := Ada.Strings.Unbounded.To_Unbounded_String ("Thick Fur");
      
    else -- Total = 6
      Creature.Integument := Special_Integument;
      
    end if;

  -- For aquatic creatures
  elsif Creature.Environment = Seashore or Creature.Swimmer       or
        Creature.Amphibious             or Creature.Fully_Aquatic then
    if Total in 1 .. 2 then
      Creature.Integument := Ada.Strings.Unbounded.To_Unbounded_String ("Thick, Sleek Fur");
      
    elsif Total = 3 then
      Creature.Integument := Ada.Strings.Unbounded.To_Unbounded_String ("Scales");
      
    elsif Total in 4 .. 5 then
      Creature.Integument := Ada.Strings.Unbounded.To_Unbounded_String ("Skin");
      
    else -- Total = 6
      Creature.Integument := Ada.Strings.Unbounded.To_Unbounded_String ("Mucus Over Skin");
      
    end if;

  end if;

  --------------------------------------------------------------------------------------------
  -- Generate creature's means of communication
  Total := Voice_Box_Total;
  if Total in 1 .. 2 then
    Creature.Voice_Box := Ada.Strings.Unbounded.To_Unbounded_String ("No Voice Box");
    
  elsif Total in 3 .. 5 then
    Creature.Voice_Box := Ada.Strings.Unbounded.To_Unbounded_String ("Grunts, Purrs and Howls only (emotions only)");
    
  else -- Total >= 6
    Total              := Voice_Box_Total;
    Creature.Voice_Box := Ada.Strings.Unbounded.To_Unbounded_String ("Ordinary Voice Box; ");
    if Total = 1 then
      Creature.Voice_Box := C (Creature.Voice_Box, "Very Low Frequency Range (subsonic)");
    elsif Total in 3 .. 5 then
      Creature.Voice_Box := C (Creature.Voice_Box, "Normal Frequency Range (human-like)");
    elsif Total in 6 .. 7 then
      Creature.Voice_Box := C (Creature.Voice_Box, "Very High Frequency Range (ultrasonic)");
    else -- Total >= 8
      Creature.Voice_Box := C (Creature.Voice_Box, "Extraordinary Frequency Range");
    end if;
    
  end if;
  
  --------------------------------------------------------------------------------------------
  -- Generate any special means of communication the creature has
  Total := Two_D;
  if Creature.Diet = Ergivore then
    Total := Total + 2;
  end if;
  if Creature.Gregariousness > 0 then
    Total := Total + 1;
  end if;

  if Total <= 9 then
    null;
  elsif Total in 10 .. 11 then
    Creature.Special_Comms := Ada.Strings.Unbounded.To_Unbounded_String ("Pheromones or other chemical methods");
  elsif Total = 12 then
    Creature.Special_Comms := Ada.Strings.Unbounded.To_Unbounded_String ("Colour patterns or cillia flickers");
  else -- Total >= 13
    Creature.Special_Comms := Ada.Strings.Unbounded.To_Unbounded_String ("Radio frequency communications");
  end if;
  
  --------------------------------------------------------------------------------------------
  -- Generate creature's vision
  if Creature.Unusual_Env = Burrower then
    Total := One_D;
    if Total = 1 then
      Creature.Vision      := Ada.Strings.Unbounded.To_Unbounded_String ("Virtually Blind (can sense the presence of light but nothing else)");
      Creature.Vision_Done := True;
    end if;
  end if;

  if not Creature.Vision_Done then
    Total := One_D;
    
    if Creature.Swimmer or Creature.Amphibious or Creature.Fully_Aquatic then
      Total := Total - 1;
    elsif Creature.Brachiator then
      Total := Total + 2;
    elsif Creature.Winged_Flight then
      Total := Total + 1;
    end if;

    if Total <= 1 then
      Creature.Vision := Ada.Strings.Unbounded.To_Unbounded_String ("Motion detection only");
      Creature.Blind  := True;
      Total := Two_D;
      if Total in 3 .. 10 then
        Creature.Curiosity := Creature.Curiosity - 1;
      end if;
      Total := Two_D;
      if Total in 3 .. 10 then
        Creature.Suspicion := Creature.Suspicion + 1;
      end if;
      
    elsif Total = 2 then
      Creature.Vision                     := Ada.Strings.Unbounded.To_Unbounded_String ("Poor");
      Creature.Num_Other                  := Creature.Num_Other     + 1;
      Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Bad Sight disadvantage");
      Total := Two_D;
      if Total in 3 .. 6 then
        Creature.Curiosity := Creature.Curiosity - 1;
      end if;
      Total := Two_D;
      if Total in 3 .. 7 then
        Creature.Suspicion := Creature.Suspicion + 1;
      end if;
      
    elsif Total in 3 .. 4 then
      Creature.Vision                     := Ada.Strings.Unbounded.To_Unbounded_String ("Average");
      Creature.Num_Other                  := Creature.Num_Other     + 1;
      Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Poor Vision; -2 to sight rolls");
      
    elsif Total = 5 then
      Creature.Vision := Ada.Strings.Unbounded.To_Unbounded_String ("Good (human range)");
      
    elsif Total = 6 then
      Creature.Vision := Ada.Strings.Unbounded.To_Unbounded_String ("Excellent");
      Creature.Num_Other      := Creature.Num_Other     + 1;
      Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Acute Vision +2");
      
    elsif Total = 7 then
      Creature.Vision := Ada.Strings.Unbounded.To_Unbounded_String ("Extraordinary (hawk-like)");
      Creature.Num_Other      := Creature.Num_Other     + 1;
      Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Acute Vision +5");
      Creature.Num_Other      := Creature.Num_Other     + 1;
      Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("+1 with Ranged Attacks");
      
    else -- Total >= 8
      Creature.Vision := Ada.Strings.Unbounded.To_Unbounded_String ("Astonishing");
      Creature.Num_Other      := Creature.Num_Other     + 1;
      Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Acute Vision +8");
      Creature.Num_Other      := Creature.Num_Other     + 1;
      Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("+2 with Ranged Attacks");
      
    end if;
  end if;
  
  -- Get number of eyes
  Total := One_D;
  if Total = 1 then
    Total := One_D;
    if Total = 1 then
      Creature.Number_Of_eyes := 1;
      
    elsif Total in 2 .. 4 then
      Creature.Number_Of_eyes := 2;
      
    elsif Total = 5 then
      Creature.Number_Of_eyes := 3;
      
    else -- Total = 6
      Creature.Number_Of_eyes := Two_D;
      
    end if;
    
  elsif Total in 2 .. 4 then
    Creature.Number_Of_eyes := 2;
    
  elsif Total = 5 then
    Creature.Number_Of_eyes := 3;
    
  else -- Total = 6
    Creature.Number_Of_eyes := Two_D;
    
  end if;

  -- Get eye placement
  Total := One_D;
  if Creature.Chain_Posn = Low then
    Total := Total + 1;
  elsif Creature.Chain_Posn = Top_Shared or Creature.Chain_Posn = Top_Solitary then
    Total := Total - 1;
  end if;
  if Creature.Diet = Carnivore then
    Total := Total - 1;
  elsif Creature.Diet = Herbivore then
    Total := Total + 1;
  end if;
  if Creature.Number_Of_Eyes >= 6 then
    Total := Total + 2;
  elsif Creature.Number_Of_Eyes >= 3 then
    Total := Total + 1;
  end if;

  if Total <= 1 then
    Creature.Eye_Placement := Ada.Strings.Unbounded.To_Unbounded_String ("Narrow Field of View; has Binocular Vision but cannot have Peripheral Vision");
    
  elsif Total in 2 .. 3 then
    Creature.Eye_Placement := Ada.Strings.Unbounded.To_Unbounded_String ("Front (as for humans); has Binocular Vision");
    
  elsif Total in 4 .. 5 then
    Creature.Eye_Placement := Ada.Strings.Unbounded.To_Unbounded_String ("Wide; has Peripheral Vision and Non-Binocular Vision");
    
  elsif Total = 6 then
    Creature.Eye_Placement := Ada.Strings.Unbounded.To_Unbounded_String ("Eyes on Stalks; has Peripheral Vision and can look around corners");
    
  else -- Total >= 7
    if Creature.Number_Of_Eyes < 4 then
      Creature.Eye_Placement := Ada.Strings.Unbounded.To_Unbounded_String ("All-Around 360 Degree Vision and Non-Binocular Vision");
    else
      Creature.Eye_Placement := Ada.Strings.Unbounded.To_Unbounded_String ("All-Around 360 Degree Vision");
    end if;
  end if;

  -- Get any special visual powers
  for Ctr in 1 .. 2 loop
    Total := Three_D;
    if Creature.Diet = Carnivore then
      Total := Total - 2;
    elsif Creature.Diet = Ergivore then
      Total := Total + 2;
    end if;
    
    if Total <= 1 then
      Creature.Special_Vision (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Spectrum Vision");
      
    elsif Total in 2 .. 3 then
      Creature.Special_Vision (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Infravision");
      
    elsif Total = 4 then
      Creature.Special_Vision (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Night Vision");
      
    elsif Total in 5 .. 6 then
      Creature.Special_Vision (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Nictating Membrance");
      
    elsif Total = 7 then
      Creature.Special_Vision (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Colour-Blind");
      
    elsif Total in 8 .. 12 then
      null; -- No special vision power
      
    elsif Total = 13 then
      Creature.Special_Vision (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Colour-Blind");
      
    elsif Total = 14 then
      Creature.Special_Vision (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Telescopic Vision");
      
    elsif Total = 15 then
      Creature.Special_Vision (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Polarized Eyes");
      
    elsif Total = 16 then
      Creature.Special_Vision (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Independantly Focusable Eyes");
      
    elsif Total = 17 then
      Creature.Special_Vision (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Spectrum Vision");
      
    else -- Total >= 18
      Creature.Special_Vision (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Field Sense");
      
    end if;
    
  end loop;

  -- Remove duplicates
  if Creature.Special_Vision (1) = Creature.Special_Vision (2) then
    Creature.Special_Vision (2) := Ada.Strings.Unbounded.Null_Unbounded_String;
  end if;

  --------------------------------------------------------------------------------------------
  -- Generate creature's hearing
  Total := One_D;
  if Total = 1 then
    Creature.Hearing                    := Ada.Strings.Unbounded.To_Unbounded_String ("Poor");
    Creature.Num_Other                  := Creature.Num_Other     + 1;
    Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Hard of Hearing");
    
  elsif Total in 2 .. 3 then
    Creature.Hearing                    := Ada.Strings.Unbounded.To_Unbounded_String ("Average (human-level)");
    
  elsif Total in 4 .. 5 then
    Creature.Hearing                    := Ada.Strings.Unbounded.To_Unbounded_String ("Good (dog-like)");
    Creature.Num_Other                  := Creature.Num_Other     + 1;
    Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Acute Hearing +2");
    
  else -- Total = 6
    Creature.Hearing                    := Ada.Strings.Unbounded.To_Unbounded_String ("Excellent");
    Creature.Num_Other                  := Creature.Num_Other     + 1;
    Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Acute Hearing +5");
    
  end if;

  -- Get any special hearing powers
  for Ctr in 1 .. 2 loop
    Total := One_D;
    if Creature.Amphibious or Creature.Fully_Aquatic then
      Total := Total + 1;
    end if;
    if Creature.Blind then
      Total := Total + 2;
    end if;

    if Total <= 1 then
      null;
      
    elsif Total in 2 .. 3 then
      Creature.Special_Hearing (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Directional Hearing (as humans)");
      
    elsif Total = 4 then
      Creature.Special_Hearing (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Hears into the Subsonic");
      
    elsif Total = 5 then
      Creature.Special_Hearing (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Hears into the Ultrasonic");
      
    else -- Total >= 6
      Creature.Special_Hearing (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Sonar (as dolphins or bats)");
      
    end if;
    
  end loop;

  -- Remove duplicates
  if Creature.Special_Hearing (1) = Creature.Special_Hearing (2) then
    Creature.Special_Hearing (2) := Ada.Strings.Unbounded.Null_Unbounded_String;
  end if;

  --------------------------------------------------------------------------------------------
  -- Generate creature's kinesthetic senses
  Total := One_D;
  if Creature.Brachiator then
    Total := Total + 1;
  elsif Creature.Winged_Flight then
    Total := Total + 2;
  end if;
  if Creature.Size = Small then
    Total := Total + 1;
  elsif Creature.Size = Huge or Creature.Size = Giant then
    Total := Total - 2;
  end if;

  if Total <= 2 then
    Creature.Kinesthetic   := Ada.Strings.Unbounded.To_Unbounded_String ("Poor");
    Creature.Kinesthetic_2 := Poor;
    Creature.Dx            := Creature.Dx - 1;
    
  elsif Total in 3 .. 4 then
    Creature.Kinesthetic   := Ada.Strings.Unbounded.To_Unbounded_String ("Average (human-level)");
    Creature.Kinesthetic_2 := Average;
    
  elsif Total = 5 then
    Creature.Kinesthetic   := Ada.Strings.Unbounded.To_Unbounded_String ("Good (as cats or monkeys)");
    Creature.Kinesthetic_2 := Good;
    Creature.Num_Other     := Creature.Num_Other     + 1;
    Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Perfect Balance");
    
  else -- Total >= 6
    Creature.Kinesthetic   := Ada.Strings.Unbounded.To_Unbounded_String ("Excellent (born acrobats)");
    Creature.Kinesthetic_2 := Excellent;
    Creature.Dx            := Creature.Dx + 2;
    Creature.Num_Other     := Creature.Num_Other     + 1;
    Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Perfect Balance");
    
  end if;

  -- Get any special kinesthic abilities
  Total := Two_D;
  if Creature.Environment = Desert or Creature.Secondary_Env = Desert or
     Creature.Unusual_Env = Ice_Desert                                then
    Total := Total + 1;
  elsif Creature.Fully_Aquatic then
    Total := Total - 1;
  end if;

  if Total <= 2 then
    Creature.Special_Kines := Ada.Strings.Unbounded.To_Unbounded_String ("Full Coordination");
    
  elsif Total in 3 .. 8 then
    null;
    
  elsif Total = 9 then
    Creature.Special_Kines := Ada.Strings.Unbounded.To_Unbounded_String ("Can (or must) hibernate under certain conditions; this is Metabolism Control with Limitations");
    
  elsif Total = 10 then
    Creature.Special_Kines := Ada.Strings.Unbounded.To_Unbounded_String ("Metabolism Control; ");
    case Creature.Kinesthetic_2 is
      when None =>
        Creature.Special_Kines := C (Creature.Special_Kines, "Power 1");
      when Poor =>
        Creature.Special_Kines := C (Creature.Special_Kines, "Power 2");
      when Average =>
        Creature.Special_Kines := C (Creature.Special_Kines, "Power 3");
      when Good =>
        Creature.Special_Kines := C (Creature.Special_Kines, "Power 4");
      when Excellent =>
        Creature.Special_Kines := C (Creature.Special_Kines, "Power 8");
    end case;
    
  elsif Total = 11 then
    Creature.Special_Kines := Ada.Strings.Unbounded.To_Unbounded_String ("Species can guide its healing processes (Rapid Healing)");
    
  else -- Total >= 12
    Creature.Special_Kines := Ada.Strings.Unbounded.To_Unbounded_String ("Rapid Healing and Metabolism Control; can control own immune systems and more; increase HT by IQ/4 for fighting diease and healing");
    
  end if;

  --------------------------------------------------------------------------------------------
  -- Generate creature's sense of taste
  Total := One_D;
  if Creature.Blind then
    Total := Total + 1;
  end if;
  if Creature.Diet = Herbivore then
    Total := Total + 1;
  end if;

  if Total <= 1 then
    Creature.Taste := Ada.Strings.Unbounded.To_Unbounded_String ("No sense of Taste");
    
  elsif Total = 2 then
    Creature.Taste                      := Ada.Strings.Unbounded.To_Unbounded_String ("Poor");
    Creature.Num_Other                  := Creature.Num_Other     + 1;
    Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Poor Sense of Taste -4");
    
  elsif Total in 3 .. 4 then
    Creature.Taste := Ada.Strings.Unbounded.To_Unbounded_String ("Average (as human)");
    
  elsif Total in 5 .. 6 then
    Creature.Taste                      := Ada.Strings.Unbounded.To_Unbounded_String ("Good");
    Creature.Num_Other                  := Creature.Num_Other     + 1;
    Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Acute Sense of Taste +2");
    
  else -- Total >= 7
    Creature.Taste                      := Ada.Strings.Unbounded.To_Unbounded_String ("Excellent; with training can determine exact chemical makeup of substances");
    Creature.Num_Other                  := Creature.Num_Other     + 1;
    Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Acute Sense of Taste +5");
    
  end if;

  --------------------------------------------------------------------------------------------
  -- Generate creature's sense of smell
  Total := One_D;
  if Creature.Blind then
    Total := Total + 1;
  end if;
  if Creature.Diet = Carnivore then
    Total := Total + 1;
  end if;

  if Total <= 1 then
    Creature.Smell := Ada.Strings.Unbounded.To_Unbounded_String ("No sense of Smell");
    
  elsif Total = 2 then
    Creature.Smell                      := Ada.Strings.Unbounded.To_Unbounded_String ("Poor");
    Creature.Num_Other                  := Creature.Num_Other     + 1;
    Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Poor Sense of Smell -4");
    
  elsif Total in 3 .. 4 then
    Creature.Smell := Ada.Strings.Unbounded.To_Unbounded_String ("Average (as human)");
    
  elsif Total in 5 .. 6 then
    Creature.Smell                      := Ada.Strings.Unbounded.To_Unbounded_String ("Good; can track by scent");
    Creature.Num_Other                  := Creature.Num_Other     + 1;
    Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Acute Sense of Smell +2");
    
  else -- Total >= 7
    Creature.Smell                      := Ada.Strings.Unbounded.To_Unbounded_String ("Excellent; Bloodhound level; can track by scent and detect and identify very weak scents");
    Creature.Num_Other                  := Creature.Num_Other     + 1;
    Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Acute Sense of Smell +5");
    
    Total := Two_D;
    if Total in 3 .. 9 then
      Creature.Empathy := Creature.Empathy + 1;
    end if;
    
  end if;

  --------------------------------------------------------------------------------------------
  -- Generate creature's intelligence and sentience
  Total := One_D;
  
  if Total = 1 then
    Creature.Iq          := 5;
    Creature.Presentient := True;
    
  elsif Total = 2 then
    Creature.Iq          := 6;
    
  elsif Total in 3 .. 4 then
    Creature.Iq          := 6;
    Creature.Presentient := True;
    
  elsif Total = 5 then
    Creature.Iq          := 7;
    
  else -- Total = 6
    Creature.Iq          := 7;
    Creature.Presentient := True;
    
  end if;

  --------------------------------------------------------------------------------------------
  -- Generate creature's special abilities (if any)
  Total := Two_D;
  
  if Total = 12 then
    Total := One_D;
    if Total = 1 then
      Creature.Special_Ability := Ada.Strings.Unbounded.To_Unbounded_String ("Absolute Timing");
      
    elsif Total = 2 then
      Creature.Special_Ability := Ada.Strings.Unbounded.To_Unbounded_String ("Intuition");
      
    elsif Total = 3 then
      Creature.Special_Ability := Ada.Strings.Unbounded.To_Unbounded_String ("Lightning Calculator");
      
    elsif Total = 4 then
      Creature.Special_Ability := Ada.Strings.Unbounded.To_Unbounded_String ("Eidetic Memory 1");
      
    elsif Total = 5 then
      Creature.Special_Ability := Ada.Strings.Unbounded.To_Unbounded_String ("Absolute Direction");
      
    else -- Total = 6
      Total := Three_D;
      
      if Creature.Feeding_Method = Trapper then
        Total := Total + 1;
      end if;
      if Creature.Chain_Posn = Low then
        Total := Total - 3;
      elsif Creature.Chain_Posn = Top_Shared or Creature.Chain_Posn = Top_Solitary then
        Total := Total + 3;
      end if;
      if Creature.Society = Pack_Troop then
        Total := Total - 1;
      elsif Creature.Society = Herd then
        Total := Total - 2;
      end if;
      
      if Total <= 1 then
        Creature.Special_Ability := Ada.Strings.Unbounded.To_Unbounded_String ("Telesend and Telereceive power 6, Psi Shield power 3");
        
      elsif Total = 2 then
        Creature.Special_Ability := Ada.Strings.Unbounded.To_Unbounded_String ("Telesend power 8, Psi Shield power 4");
        
      elsif Total = 3 then
        Creature.Special_Ability := Ada.Strings.Unbounded.To_Unbounded_String ("Telereceive power 8, Psi Shield power 4");
        
      elsif Total in 4 .. 5 then
        Creature.Special_Ability := Ada.Strings.Unbounded.To_Unbounded_String ("Precognition power 6, Psi Shield power 3");
        
      elsif Total = 6 then
        Creature.Special_Ability := Ada.Strings.Unbounded.To_Unbounded_String ("Clairvoyance/Clairaudience power 6, Psi Shield power 3");
        
      elsif Total in 7 .. 9 then
        Creature.Special_Ability := Ada.Strings.Unbounded.To_Unbounded_String ("Emotion Sense power 8");
        
      elsif Total in 10 .. 12 then
        Creature.Special_Ability := Ada.Strings.Unbounded.To_Unbounded_String ("Danger Sense");
        
      elsif Total = 13 then
        Creature.Special_Ability := Ada.Strings.Unbounded.To_Unbounded_String ("Psionic Resistance power 10");
        
      elsif Total = 14 then
        Creature.Special_Ability := Ada.Strings.Unbounded.To_Unbounded_String ("Luck, Psi Shield power 6");
        
      elsif Total = 15 then
        Creature.Special_Ability := Ada.Strings.Unbounded.To_Unbounded_String ("Psi Static power 10");
        
      elsif Total = 16 then
        Creature.Special_Ability := Ada.Strings.Unbounded.To_Unbounded_String ("Telekinesis power 8");
        
      elsif Total = 17 then
        Creature.Special_Ability := Ada.Strings.Unbounded.To_Unbounded_String ("Telesend power 6");
        
      elsif Total in 18 .. 19 then
        Creature.Special_Ability := Ada.Strings.Unbounded.To_Unbounded_String ("Sleep power 6, Psi Shield power 3");
        
      elsif Total = 20 then
        Creature.Special_Ability := Ada.Strings.Unbounded.To_Unbounded_String ("Mental Blow power 10, Psi Shield power 5");
        
      else -- Total >= 21
        Creature.Special_Ability := Ada.Strings.Unbounded.To_Unbounded_String ("Telereceive and Telecontrol power 8, Psi Shield power 4");
        
      end if;
      
    end if;
    
  end if;

  --------------------------------------------------------------------------------------------
  -- Generate variations in creature's abilities
  Creature.St              := Modified_Attribute (Creature.St);
  Creature.Dx              := Modified_Attribute (Creature.Dx);
  Creature.Ht              := Modified_Attribute (Creature.Ht);

  Creature.Chauvinism      := Modified_Attribute (Creature.Chauvinism);
  Creature.Concentration   := Modified_Attribute (Creature.Concentration);
  Creature.Curiosity       := Modified_Attribute (Creature.Curiosity);
  Creature.Egotism         := Modified_Attribute (Creature.Egotism);
  Creature.Empathy         := Modified_Attribute (Creature.Empathy);
  Creature.Gregariousness  := Modified_Attribute (Creature.Gregariousness);
  Creature.Imagination     := Modified_Attribute (Creature.Imagination);
  Creature.Suspicion       := Modified_Attribute (Creature.Suspicion);

  --------------------------------------------------------------------------------------------
  -- Determine creature's personailty traits
  
  if Creature.Chauvinism >= 3 then
    if Creature.Suspicion > 1 then
      Creature.Chauv_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Xenophobia");
    else
      Creature.Chauv_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Chauvinistic");
    end if;
    
  elsif Creature.Chauvinism = 2 then
    if Creature.Empathy < 1 or Creature.Suspicion > -1 then
      Creature.Chauv_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Racial Intolerance");
    else
      Creature.Chauv_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Chauvinistic");
    end if;
    
  elsif Creature.Chauvinism = 1 then
    if Creature.Empathy < 0 or Creature.Suspicion > 0 then
      Creature.Chauv_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Racial Intolerance");
    else
      Creature.Chauv_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Chauvinistic (Quirk)");
    end if;
    
  elsif Creature.Chauvinism = 0 then
    null;
    
  elsif Creature.Chauvinism = -1 then
    Creature.Chauv_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Broad-Minded (Quirk)");
    
  elsif Creature.Chauvinism = -2 then
    if Creature.Empathy > 0 or Creature.Suspicion < 0 then
      Creature.Chauv_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Xenophilia");
    else
      Creature.Chauv_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Broad-Minded (Quirk)");
    end if;
    
  else -- Creature.Chauvinism <= -3
    if (Creature.Empathy > 0 and Creature.Suspicion >= 0) or
       (Creature.Empathy <= 0 and Creature.Suspicion < 0) then
      Creature.Chauv_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Xenophilia");
    elsif Creature.Empathy > 0 and Creature.Suspicion < 0 then
      Creature.Chauv_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Strong Xenophilia");
    else
      Creature.Chauv_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Undiscriminating (Quirk)");
    end if;
    
  end if;

  --------------------------------------------------------------------------------------------
  if Creature.Concentration >= 3 then
    Creature.Conc_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Single-Minded and has ");
    Total := D_4;
    if Total = 1 then
      Creature.Conc_Trait := C (Creature.Conc_Trait, "High Pain Threshold");
    elsif Total = 2 then
      Creature.Conc_Trait := C (Creature.Conc_Trait, "Mathematical Ability");
    elsif Total = 3 then
      Creature.Conc_Trait := C (Creature.Conc_Trait, "Musical Ability");
    else -- Total = 4
      Creature.Conc_Trait := C (Creature.Conc_Trait, "Language Talent");
    end if;
    
  elsif Creature.Concentration = 2 then
    Creature.Conc_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Single-Minded");
    
  elsif Creature.Concentration = 1 then
    Creature.Conc_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Attentive (Quirk)");
    
  elsif Creature.Concentration = 0 then
    null;
    
  elsif Creature.Concentration = -1 then
    Creature.Conc_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Distractible (Quirk)");
    
  elsif Creature.Concentration = -2 then
    Creature.Conc_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Short Attention Span");
    
  else -- Creature.Concentration <= -3
    Creature.Conc_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Absent-Minded");
    
  end if;

  --------------------------------------------------------------------------------------------
  if Creature.Curiosity >= 3 then
    Creature.Curio_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Very Curious");
    if Creature.Concentration <= 0 or Creature.Suspicion <= 0 then
      Creature.Curio_Trait := C (Creature.Curio_Trait, "; roll at -2 to resist");
    end if;
    
  elsif Creature.Curiosity = 2 then
    Creature.Curio_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Curious");
    if Creature.Concentration <= 0 then
      Creature.Curio_Trait := C (Creature.Curio_Trait, "; roll at -2 to resist");
    end if;
    
  elsif Creature.Curiosity = 1 then
    if Creature.Concentration <= 0 then
      Creature.Curio_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Curious");
    else
      Creature.Curio_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Nosy (Quirk)");
    end if;
    
  elsif Creature.Curiosity = 0 then
    null;
    
  elsif Creature.Curiosity = -1 then
    Creature.Curio_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Staid (Quirk)");
    
  elsif Creature.Curiosity = -2 then
    if Creature.Suspicion <= 0 then
      Creature.Curio_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Obdurate");
    else
      Creature.Curio_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Incurious");
    end if;
    
  else -- Creature.Curiosity <= -3
    Creature.Curio_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Obdurate");
    
  end if;

  --------------------------------------------------------------------------------------------
  if not Creature.Individ_Queens then
    Creature.Ego_Trait := Determine_Egotism (Egotism    => Creature.Egotism,
                                             Empathy    => Creature.Empathy, 
                                             Chauvinism => Creature.Chauvinism, 
                                             Suspicion  => Creature.Suspicion);
  else
    Creature.Ego_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Queens are ") &
                          Determine_Egotism (Egotism    => Creature.Egotism + 2,
                                             Empathy    => Creature.Empathy, 
                                             Chauvinism => Creature.Chauvinism, 
                                             Suspicion  => Creature.Suspicion) &
                          Ada.Strings.Unbounded.To_Unbounded_String ("; others are ") &
                          Determine_Egotism (Egotism    => Creature.Egotism - 1,
                                             Empathy    => Creature.Empathy, 
                                             Chauvinism => Creature.Chauvinism, 
                                             Suspicion  => Creature.Suspicion);
  end if;

  --------------------------------------------------------------------------------------------
  if Creature.Empathy >= 3 then
    Creature.Emp_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Empathy (Advantage)");
    if Creature.Gregariousness > 0 then
      Creature.Emp_Trait := C (Creature.Emp_Trait, " and Charitable");
    end if;
    
  elsif Creature.Empathy = 2 then
    Creature.Emp_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Sensitive");
    if Creature.Gregariousness > 1 then
      if Three_D <= 8 then
        Creature.Emp_Trait := C (Creature.Emp_Trait, " and Charitable");
      end if;
    end if;
    
  elsif Creature.Empathy = 1 then
    if Creature.Gregariousness > 0 and Creature.Suspicion < 0 then
      Creature.Emp_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Sensitive");
    else
      Creature.Emp_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Responsive (Quirk)");
    end if;
    
  elsif Creature.Empathy = 0 then
    null;
    
  elsif Creature.Empathy = -1 then
    Creature.Emp_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Oblivious (Quirk)");
    
  elsif Creature.Empathy = -2 then
    Creature.Emp_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Callous");
    if Creature.Diet = Carnivore then
      if One_D = 1 then
        Creature.Emp_Trait := C (Creature.Emp_Trait, " and has Bloodlust");
      end if;
    end if;
    
  else -- Creature.Empathy <= -3
    Creature.Emp_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Solipsist");
    if Creature.Diet = Carnivore then
      if One_D <= 3 then
        Creature.Emp_Trait := C (Creature.Emp_Trait, " and has Bloodlust");
      end if;
    end if;
    
  end if;

  --------------------------------------------------------------------------------------------
  if Creature.Gregariousness >= 3 then
    Creature.Gregar_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Gregarious");
    
  elsif Creature.Gregariousness = 2 then
    Creature.Gregar_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Chummy");
    
  elsif Creature.Gregariousness = 1 then
    Creature.Gregar_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Congenial (Quirk)");
    
  elsif Creature.Gregariousness = 0 then
    null;
    
  elsif Creature.Gregariousness = -1 then
    Creature.Gregar_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Uncongenial (Quirk)");
    
  elsif Creature.Gregariousness = -2 then
    Creature.Gregar_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Loner");
    
  else -- Creature.Gregariousness <= -3
    Creature.Gregar_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Reclusive");
    
  end if;

  --------------------------------------------------------------------------------------------
  if Creature.Imagination >= 3 then
    if Creature.Concentration > 0 and Creature.Egotism < 2 then
      Creature.Imag_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Versatile");
    else
      Creature.Imag_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Imaginative (Quirk)");
    end if;
    
    if Creature.Empathy < 1 then
      Creature.Imag_Trait := C (Creature.Imag_Trait, " and Odious Racial Habit (Won't shut up about their great ideas) -5 points");
    end if;
    
  elsif Creature.Imagination = 2 then
    if Creature.Concentration > 0 and Creature.Egotism < 2 then
      Creature.Imag_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Versatile");
    else
      Creature.Imag_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Imaginative (Quirk)");
    end if;
    
    if Creature.Egotism > 0 or Creature.Concentration < 1 then
      Creature.Imag_Trait := C (Creature.Imag_Trait, " and Dreamer (Quirk)");
    end if;
    
  elsif Creature.Imagination = 1 then
    if Creature.Concentration > 0 and Creature.Egotism < 2 then
      Creature.Imag_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Versatile");
    else
      Creature.Imag_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Imaginative (Quirk)");
    end if;
    
  elsif Creature.Imagination = 0 then
    null;
    
  elsif Creature.Imagination = -1 then
    Creature.Imag_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Dull (Quirk)");
    
  elsif Creature.Imagination = -2 then
    Creature.Imag_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Hidebound");
    
  else -- Creature.Imagination <= -3
    Creature.Imag_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Hidebound");
    Creature.Iq         := Creature.Iq - 1;
    
  end if;

  --------------------------------------------------------------------------------------------
  if Creature.Suspicion >= 3 then
    Creature.Susp_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Edgy");
    
    Total := Three_D;
    if Creature.Diet = Carnivore then
      Total := Total + 3;
    elsif Creature.Diet = Herbivore then
      Total := Total - 3;
    end if;
    if Total < 10 then
      Creature.Susp_Trait := C (Creature.Susp_Trait, " and Cowardice");
    else -- Total >= 10
      Creature.Susp_Trait := C (Creature.Susp_Trait, " and Paranoid");
    end if;
    
  elsif Creature.Suspicion = 2 then
    if Creature.Curiosity <= -3 then
      Creature.Susp_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Careful (Quirk)");
    else
      Creature.Susp_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Edgy");
    end if;
    
  elsif Creature.Suspicion = 1 then
    if Creature.Curiosity <= -2 then
      null;
    else
      Creature.Susp_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Careful (Quirk)");
    end if;
    
  elsif Creature.Suspicion = 0 then
    null;
    
  elsif Creature.Suspicion = -1 then
    Creature.Susp_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Cool");
    
  elsif Creature.Suspicion = -2 then
    Creature.Susp_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Collected");
    if Creature.Egotism >= 2 then
      Creature.Susp_Trait := C (Creature.Imag_Trait, " and Overconfident");
    end if;
    
  else -- Creature.Suspicion <= -3
    Creature.Susp_Trait := Ada.Strings.Unbounded.To_Unbounded_String ("Imperturbable");
    if Creature.Egotism >= 1 then
      Creature.Susp_Trait := C (Creature.Susp_Trait, " and Overconfident");
    end if;
    
  end if;

  --------------------------------------------------------------------------------------------
  -- Determine creature's 'Family Ties'
  Total := Three_D;
  if Total = 3 then
    Total := One_D;
    if Total = 1 then
      Creature.Social_Standing := Ada.Strings.Unbounded.To_Unbounded_String ("Major Race (Tandu, Soro or Lesh level)");
    elsif Total in 2 .. 3 then
      Creature.Social_Standing := Ada.Strings.Unbounded.To_Unbounded_String ("Powerful and well-known species");
    elsif Total in 4 .. 5 then
      Creature.Social_Standing := Ada.Strings.Unbounded.To_Unbounded_String ("Species powerful in its neighbourhood");
    else -- Total = 6
      Creature.Social_Standing            := Ada.Strings.Unbounded.To_Unbounded_String ("Retired Race");
      Creature.Num_Other                  := Creature.Num_Other     + 1;
      Creature.Other (Creature.Num_Other) := Ada.Strings.Unbounded.To_Unbounded_String ("Racial Status +1");
    end if;
    
  elsif Total in 4 .. 5 then
    Creature.Social_Standing := Ada.Strings.Unbounded.To_Unbounded_String ("Notable Patron Race");
    
  elsif Total = 6 then
    Creature.Social_Standing := Ada.Strings.Unbounded.To_Unbounded_String ("Average Patron Race");
    
  elsif Total in 7 .. 8 then
    Creature.Social_Standing := Ada.Strings.Unbounded.To_Unbounded_String ("Minor Patron Race");
    
  elsif Total = 9 then
    Creature.Social_Standing := Ada.Strings.Unbounded.To_Unbounded_String ("Powerful Client Race");
    
  elsif Total in 10 .. 13 then
    Creature.Social_Standing := Ada.Strings.Unbounded.To_Unbounded_String ("Average Client Race");
    
  elsif Total in 14 .. 16 then
    Creature.Social_Standing := Ada.Strings.Unbounded.To_Unbounded_String ("Minor Client Race");
    
  else -- Total in 17 .. 18
    Creature.Social_Standing := Ada.Strings.Unbounded.To_Unbounded_String ("New, Weird or Botched Client Race");
    
  end if;
  
  for Ctr in 1 .. 3 loop
    Total := Three_D;
    if Total in 3 .. 4 then
      Creature.Special_Cases (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Bad Reputation");
      Br (Ctr) := Ctr;
      
    elsif Total = 5 then
      Creature.Special_Cases (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Influential in Minor Insititute; can take as Patron for half cost");
      Mi (Ctr) := Ctr;
      
    elsif Total = 6 then
      Creature.Special_Cases (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Member of Major Religious Alliance; can take as Patron for half cost");
      
    elsif Total = 7 then
      Creature.Special_Cases (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Race is Destitue or Suffering from Fines");
      Creature.Wealth              := Ada.Strings.Unbounded.To_Unbounded_String ("Racial Wealth: Struggling");
      
    elsif Total in 8 .. 13 then
      null; -- Nothing special
      
    elsif Total = 14 then
      Creature.Special_Cases (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Race is a local military power");
      Lmp (Ctr)                    := Ctr;
      
    elsif Total = 15 then
      Creature.Special_Cases (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Race runs a trade cartel; can take as Patron for half cost");
      Creature.Wealth              := Ada.Strings.Unbounded.To_Unbounded_String ("Racial Wealth: Comfortable");
      
    elsif Total = 16 then
      Creature.Special_Cases (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Race controls a minor transfer point");
      Creature.Wealth              := Ada.Strings.Unbounded.To_Unbounded_String ("Racial Wealth: Comfortable");
      Ctp (Ctr)                    := Ctr;
      
    else -- Total in 17 .. 18
      Creature.Special_Cases (Ctr) := Ada.Strings.Unbounded.To_Unbounded_String ("Unusual biochemistry; race can only live on certain rare worlds");
      
    end if;
    
  end loop;

  if Br (1) > 0 and Br (2) > 0 and Br (3) = 0 then
    Creature.Special_Cases (1) := Ada.Strings.Unbounded.To_Unbounded_String ("Race is a Wolfling Race!");
    Creature.Special_Cases (2) := Ada.Strings.Unbounded.Null_Unbounded_String;
    
  elsif Br (1) > 0 and Br (2) = 0 and Br (3) > 0 then
    Creature.Special_Cases (1) := Ada.Strings.Unbounded.To_Unbounded_String ("Race is a Wolfling Race!");
    Creature.Special_Cases (3) := Ada.Strings.Unbounded.Null_Unbounded_String;
    
  elsif Br (1) = 0 and Br (2) > 0 and Br (3) > 0 then
    Creature.Special_Cases (2) := Ada.Strings.Unbounded.To_Unbounded_String ("Race is a Wolfling Race!");
    Creature.Special_Cases (3) := Ada.Strings.Unbounded.Null_Unbounded_String;
    
  elsif Br (1) > 0 and Br (2) > 0 and Br (3) > 0 then
    Creature.Special_Cases (1) := Ada.Strings.Unbounded.To_Unbounded_String ("Race is a Wolfling Race!");
    Creature.Special_Cases (2) := Ada.Strings.Unbounded.Null_Unbounded_String;
    Creature.Special_Cases (3) := Ada.Strings.Unbounded.Null_Unbounded_String;
    
  end if;
  
  if Ctp (1) > 0 and Ctp (2) > 0 and Ctp (3) = 0 then
    Creature.Special_Cases (1) := Ada.Strings.Unbounded.To_Unbounded_String ("Race controls a major transfer point");
    Creature.Wealth            := Ada.Strings.Unbounded.To_Unbounded_String ("Racial Wealth: Wealthy");
    
  elsif Ctp (1) > 0 and Ctp (2) = 0 and Ctp (3) > 0 then
    Creature.Special_Cases (1) := Ada.Strings.Unbounded.To_Unbounded_String ("Race controls a major transfer point");
    Creature.Wealth            := Ada.Strings.Unbounded.To_Unbounded_String ("Racial Wealth: Wealthy");
    
  elsif Ctp (1) = 0 and Ctp (2) > 0 and Ctp (3) > 0 then
    Creature.Special_Cases (2) := Ada.Strings.Unbounded.To_Unbounded_String ("Race controls a major transfer point");
    Creature.Wealth            := Ada.Strings.Unbounded.To_Unbounded_String ("Racial Wealth: Wealthy");
    
  elsif Ctp (1) > 0 and Ctp (2) > 0 and Ctp (3) > 0 then
    Creature.Special_Cases (1) := Ada.Strings.Unbounded.To_Unbounded_String ("Race controls a major transfer point");
    Creature.Special_Cases (2) := Ada.Strings.Unbounded.Null_Unbounded_String;
    Creature.Special_Cases (3) := Ada.Strings.Unbounded.Null_Unbounded_String;
    Creature.Wealth            := Ada.Strings.Unbounded.To_Unbounded_String ("Racial Wealth: Wealthy");
    
  end if;
  
  if (Lmp (1) > 0 and Lmp (2) > 0 and Lmp (3) = 0) then 
    Creature.Special_Cases (1) := Ada.Strings.Unbounded.To_Unbounded_String ("Race is a Regional military power");
    Creature.Special_Cases (2) := Ada.Strings.Unbounded.Null_Unbounded_String;
    
  elsif (Lmp (1) > 0 and Lmp (2) = 0 and Lmp (3) > 0) then
    Creature.Special_Cases (1) := Ada.Strings.Unbounded.To_Unbounded_String ("Race is a Regional military power");
    Creature.Special_Cases (3) := Ada.Strings.Unbounded.Null_Unbounded_String;
    
  elsif (Lmp (1) = 0 and Lmp (2) > 0 and Lmp (3) > 0) then
    Creature.Special_Cases (2) := Ada.Strings.Unbounded.To_Unbounded_String ("Race is a Regional military power");
    Creature.Special_Cases (3) := Ada.Strings.Unbounded.Null_Unbounded_String;
    
  elsif (Lmp (1) > 0 and Lmp (2) > 0 and Lmp (3) > 0) then
    Creature.Special_Cases (1) := Ada.Strings.Unbounded.To_Unbounded_String ("Race is a Galaxy-wide military power");
    Creature.Special_Cases (2) := Ada.Strings.Unbounded.Null_Unbounded_String;
    Creature.Special_Cases (3) := Ada.Strings.Unbounded.Null_Unbounded_String;
    
  end if;

  if Mi (1) > 0 and Mi (2) > 0 and Mi (3) = 0 then
    Creature.Special_Cases (1) := Ada.Strings.Unbounded.To_Unbounded_String ("Influential in Major Insititute; can take as Patron for half cost");
    Creature.Special_Cases (2) := Ada.Strings.Unbounded.Null_Unbounded_String;
    
  elsif Mi (1) > 0 and Mi (2) = 0 and Mi (3) > 0 then
    Creature.Special_Cases (1) := Ada.Strings.Unbounded.To_Unbounded_String ("Influential in Major Insititute; can take as Patron for half cost");
    Creature.Special_Cases (3) := Ada.Strings.Unbounded.Null_Unbounded_String;
    
  elsif Mi (1) = 0 and Mi (2) > 0 and Mi (3) > 0 then
    Creature.Special_Cases (2) := Ada.Strings.Unbounded.To_Unbounded_String ("Influential in Major Insititute; can take as Patron for half cost");
    Creature.Special_Cases (3) := Ada.Strings.Unbounded.Null_Unbounded_String;
    
  elsif Mi (1) > 0 and Mi (2) > 0 and Mi (3) > 0 then
    Creature.Special_Cases (1) := Ada.Strings.Unbounded.To_Unbounded_String ("Influential in Major Insititute; can take as Patron for half cost");
    Creature.Special_Cases (2) := Ada.Strings.Unbounded.Null_Unbounded_String;
    Creature.Special_Cases (3) := Ada.Strings.Unbounded.Null_Unbounded_String;

  end if;
  
  -- Remove any duplicates
  if Creature.Special_Cases (1) = Creature.Special_Cases (2) then
    Creature.Special_Cases (2) := Ada.Strings.Unbounded.Null_Unbounded_String;
  end if;
  if Creature.Special_Cases (1) = Creature.Special_Cases (3) then
    Creature.Special_Cases (3) := Ada.Strings.Unbounded.Null_Unbounded_String;
  end if;
  if Creature.Special_Cases (2) = Creature.Special_Cases (3) then
    Creature.Special_Cases (3) := Ada.Strings.Unbounded.Null_Unbounded_String;
  end if;
  
  --------------------------------------------------------------------------------------------
  -- Output the creature
  -- Could ask how many species to generate and loop through them

  -- Get Creature Name
  Ada.Text_Io.Put_Line ("Enter a name for the creature (default is " &
                        Ada.Strings.Unbounded.To_String (Creature.Name) &
                        ")");
  Creature.Name := Get_String (Creature.Name);
  
  -- Create output file
  Ada.Text_Io.Create (File => Output_File,
                      Mode => Ada.Text_Io.Out_File,
                      Name => Ada.Strings.Unbounded.To_String (Creature.Name) & ".txt",
                      Form => "");

  --------------------------------------------------------------------------------------------
  Put_Col ("UR-CREATURE NAME", Creature.Name);
  Divider;
  
  --------------------------------------------------------------------------------------------
  if Creature.Environment in Desert .. Forest then
    Put_Col ("CREATURE'S HOME ENVIRONMENT", Environments (Creature.Environment));
  elsif Creature.Environment = Seashore then
    Put_Col ("CREATURE'S HOME ENVIRONMENT",
             C (Environments (Creature.Environment), " surrounded by ") & Environments (Creature.Secondary_Env)
            );
  else -- must be Unusual_Env
    if Creature.Unusual_Env = Ice_Desert or
       Creature.Unusual_Env = Aquatic    or
       Creature.Unusual_Env = Exotic     then
      Put_Col ("CREATURE'S HOME ENVIRONMENT", Unusual_Environments (Creature.Unusual_Env));
    else
      Put_Col ("CREATURE'S HOME ENVIRONMENT",
               C (Unusual_Environments (Creature.Unusual_Env), " surrounded by ") & Environments (Creature.Secondary_Env)
              );
    end if;
  end if;
  Divider;
  
  --------------------------------------------------------------------------------------------
  Ada.Text_Io.Put_Line (Output_File, "CREATURE'S DIET");
  Put_Col ("Basic Diet",     Diets           (Creature.Diet));
  Put_Col ("Feeding Method", Feeding_Methods (Creature.Feeding_Method));
  if Creature.Feeding_Method =  Mixed then
    Put_Col ("Secondary Diet",           Diets           (Creature.Secondary_Diet));
    Put_Col ("Secondary Feeding Method", Feeding_Methods (Creature.Sec_Feed_Meth));
  end if;
  Put_Col ("Metabolism", Metabolisms (Creature.Metabolism));
  Divider;
  
  --------------------------------------------------------------------------------------------
  Put_Col ("CREATURE'S SOCIETY", Societies (Creature.Society));
  Divider;
  
  --------------------------------------------------------------------------------------------
  Ada.Text_Io.Put_Line (Output_File, "CREATURE'S SIZE AND BULK");
  Put_Col ("Creature Size",        Sizes (Creature.Size));
  Put_Col ("Creature's Build",     Bulks (Creature.Bulk));
  Put_Col ("Average Weight (lbs)", Trim (Integer'Image (Creature.Avg_Weight)));
  Divider;
  
  --------------------------------------------------------------------------------------------
  Put_Col ("CREATURE'S POSITION IN THE FOOD CHAIN", Chain_Positions (Creature.Chain_Posn));
  Divider;
  
  --------------------------------------------------------------------------------------------
  Ada.Text_Io.Put_Line (Output_File, "CREATURE'S REPRODUCTIVE STRATEGY");
  Put_Col ("Number of Offspring and Commitment to Young", Offsprings (Creature.Offspring));
  Put_Col ("Lifespan",                                    Creature.Lifespan);
  Put_Col ("Number of Sexes",                             Creature.Genders);
  Put_Col ("Method of Delivery",                          Creature.Delivery);
  Spec_Num := 0;
  for Ctr in 1 .. 2 loop
    if Creature.Reprod_Spec (Ctr) /= Ada.Strings.Unbounded.Null_Unbounded_String then
      Spec_Num := Spec_Num + 1;
      Put_Col ("Special Feature " & Trim (Integer'Image (Spec_Num)),
               Creature.Reprod_Spec (Ctr));
    end if;
  end loop;
  Divider;
  
  --------------------------------------------------------------------------------------------
  Ada.Text_Io.Put_Line (Output_File, "CREATURE'S BODY PLAN");
  if Creature.Brachiator then
    Put_Col ("Movement Method", "Brachiator (monkey-like)");
  elsif Creature.Swimmer then
    Put_Col ("Movement Method", "Swimmer (like an otter)");
  elsif Creature.Fully_Aquatic then
    Put_Col ("Movement Method", "Fully Aquatic (fish or whale-like)");
  elsif Creature.Amphibious then
    Put_Col ("Movement Method", "Amphibious (like a seal or frog)");
  elsif Creature.Glider then
    Put_Col ("Movement Method", "Glider");
  elsif Creature.Winged_Flight then
    Put_Col ("Movement Method", "Winged Flight");
  elsif Creature.Tunneller then
    Put_Col ("Movement Method", "Tunneller");
  elsif Creature.Bog_Dweller then
    Put_Col ("Movement Method", "Bog Dweller; combined walking and swimming");
  else -- must be a walker
    Put_Col ("Movement Method", "Walker");
  end if;

  Put_Col ("Creature's Symmetry",     Symmetries          (Creature.Symmetry));
  Put_Col ("Creature's Posture",      Postures            (Creature.Posture));
  Put_Col ("Number of Limbs",         Trim (Natural'Image (Creature.Num_Limbs)));
  Put_Col ("Number of Walking Limbs", Trim (Natural'Image (Creature.Walking_Limbs)));
  if Creature.Walking_Limbs > 0 then
    Put_Col ("Type of Walking Limbs", Creature.Walk_Limb_Type);
  end if;
  Put_Col ("Number of Handling Limbs", Trim (Natural'Image (Creature.Handling_Limbs)));
  if Creature.Handling_Limbs > 0 then
    Put_Col ("Type of Handling Limbs", Creature.Hand_Limb_Type);
  end if;
  if Creature.Num_Extra_Limbs /= 50 then
    Put_Col ("Number of Extra Limbs", Trim (Natural'Image (Creature.Num_Extra_Limbs)));
  else
    Put_Col ("Number of Extra Limbs", "Many");
  end if;
  if Creature.Num_Extra_Limbs > 0 then
    Put_Col ("Strength of Extra Limbs",    Creature.Ext_Limb_St);
    Put_Col ("Dexterity of Extra Limbs",   Creature.Ext_Limb_Dx);
    Put_Col ("Sensitivity of Extra Limbs", Creature.Ext_Limb_Sens);
  end if;

  Divider;
  
  --------------------------------------------------------------------------------------------
  if Creature.Natural_Weapons = Null_Repeat_Data                            and
     Creature.Natural_Poisons = Null_Repeat_Data                            and
     Creature.Extrord_Weapons = Ada.Strings.Unbounded.Null_Unbounded_String then
    -- No natural weapons
    Put_Col ("CREATURE'S NATURAL WEAPONS", "None");
  else
    Ada.Text_Io.Put_Line (Output_File, "CREATURE'S NATURAL WEAPONS");
    Spec_Num := 0;
    for Ctr in 1 .. 3 loop
      if Creature.Natural_Weapons (Ctr) /= Ada.Strings.Unbounded.Null_Unbounded_String then
        Spec_Num := Spec_Num + 1;
        Put_Col ("Natural Weapon " & Trim (Integer'Image (Spec_Num)), Creature.Natural_Weapons (Ctr));
      end if;
    end loop;
    Spec_Num := 0;
    for Ctr in 1 .. 3 loop
      if Creature.Natural_Poisons (Ctr) /= Ada.Strings.Unbounded.Null_Unbounded_String then
       Spec_Num := Spec_Num + 1;
       Put_Col ("Natural Poison " & Trim (Integer'Image (Spec_Num)),
                 C (Creature.Natural_Poisons (Ctr), " with Potency " & Trim (Natural'Image (Creature.Poison_Potency))));
      end if;
    end loop;
    if Creature.Extrord_Weapons /= Ada.Strings.Unbounded.Null_Unbounded_String then
      Put_Col ("Extraordinary Natural Weapon", Creature.Extrord_Weapons);
    end if;
  end if;
  Divider;
  
  --------------------------------------------------------------------------------------------
  Put_Col ("CREATURE'S INTEGUMENT TYPE", Creature.Integument);
  Divider;
  
  --------------------------------------------------------------------------------------------
  Ada.Text_Io.Put_Line (Output_File, "CREATURE'S MEANS OF COMMUNICATION");
  Put_Col ("Voice Box Type", Creature.Voice_Box);
  if Creature.Special_Comms /= Ada.Strings.Unbounded.Null_Unbounded_String then
    Put_Col ("Special Communications", Creature.Special_Comms);
  end if;
  Divider;
  
  --------------------------------------------------------------------------------------------
  Ada.Text_Io.Put_Line (Output_File, "CREATURE'S SENSES");
  Put_Col ("Vision",         Creature.Vision);
  Put_Col ("Number of Eyes", Trim (Natural'Image (Creature.Number_Of_Eyes)));
  Put_Col ("Eye Placement",  Creature.Eye_Placement);
  Spec_Num := 0;
  for Ctr in 1 .. 3 loop
    if Creature.Special_Vision (Ctr) /= Ada.Strings.Unbounded.Null_Unbounded_String then
      Spec_Num := Spec_Num + 1;
      Put_Col ("Special Vision Ability " & Trim (Integer'Image (Spec_Num)), Creature.Special_Vision (Ctr));
    end if;
  end loop;
  Put_Col ("Hearing", Creature.Hearing);
  Spec_Num := 0;
  for Ctr in 1 .. 3 loop
    if Creature.Special_Hearing (Ctr) /= Ada.Strings.Unbounded.Null_Unbounded_String then
      Spec_Num := Spec_Num + 1;
      Put_Col ("Special Hearing Ability " & Trim (Integer'Image (Spec_Num)), Creature.Special_Hearing (Ctr));
    end if;
  end loop;
  Put_Col ("Kinesthetic Sense", Creature.Kinesthetic);
  if Creature.Special_Kines /= Ada.Strings.Unbounded.Null_Unbounded_String then
    Put_Col ("Special Kinesthetic Ability", Creature.Special_Kines);
  end if;
  Put_Col ("Sense of Taste", Creature.Taste);
  Put_Col ("Sense of Smell", Creature.Smell);
  Divider;
 
  --------------------------------------------------------------------------------------------
  if Creature.Special_Ability /= Ada.Strings.Unbounded.Null_Unbounded_String then
    Put_Col ("CREATURE'S SPECIAL MENTAL ABILITIES", Creature.Special_Ability);
    Divider;
  end if;
  
  --------------------------------------------------------------------------------------------
  Ada.Text_Io.Put_Line (Output_File, "CREATURE'S PERSONALITY TRAITS");
  Put_Trait ("Chauvinism",     Creature.Chauv_Trait,  Creature.Chauvinism);
  Put_Trait ("Concentration",  Creature.Conc_Trait,   Creature.Concentration);
  Put_Trait ("Curiosity",      Creature.Curio_Trait,  Creature.Curiosity);
  Put_Trait ("Egotism",        Creature.Ego_Trait,    Creature.Egotism);
  Put_Trait ("Empathy",        Creature.Emp_Trait,    Creature.Empathy);
  Put_Trait ("Gregariousness", Creature.Gregar_Trait, Creature.Gregariousness);
  Put_Trait ("Imagination",    Creature.Imag_Trait,   Creature.Imagination);
  Put_Trait ("Suspicion",      Creature.Susp_Trait,   Creature.Suspicion);
  Divider;
  
  --------------------------------------------------------------------------------------------
  Ada.Text_Io.Put_Line (Output_File, "CREATURE'S STATISTICS");
  if Creature.Incr_Strength then
    Put_Col ("St", Trim (Integer'Image (Creature.St)) & " with Increased Strength");
  else
    Put_Col ("St", Trim (Integer'Image (Creature.St)));
  end if;
  Put_Col ("Dx", Trim (Integer'Image (Creature.Dx)));
  if Creature.Presentient then
    Put_Col ("Iq", Trim (Integer'Image (Creature.Iq)) & ", and Presentient");
  else
    Put_Col ("Iq", Trim (Integer'Image (Creature.Iq)));
  end if;
  Put_Col ("Ht", Trim (Integer'Image (Creature.Ht)));
  Divider;
  
  --------------------------------------------------------------------------------------------
  if Creature.Other /= Null_Other_Data then
  
    Ada.Text_Io.Put_Line (Output_File, "OTHER CHARACTERISTICS");
    for Ctr in 1 .. Creature.Num_Other loop
      Put_Col (Trim (Integer'Image (Ctr)), Creature.Other (Ctr));
    end loop;
    Divider;
    
  end if;
  
  --------------------------------------------------------------------------------------------
  Ada.Text_Io.Put_Line (Output_File, "CREATURE'S 'FAMILY TIES' (OPTIONAL)");
  Put_Col ("Galactic Social Standing", Creature.Social_Standing);
  Spec_Num := 0;
  for Ctr in 1 .. 3 loop
    if Creature.Special_Cases (Ctr) /= Ada.Strings.Unbounded.Null_Unbounded_String then
      Spec_Num := Spec_Num + 1;
      Put_Col ("Galactic Special Case " & Trim (Integer'Image (Spec_Num)), Creature.Special_Cases (Ctr));
    end if;
  end loop;
  if Creature.Wealth /= Ada.Strings.Unbounded.Null_Unbounded_String then
    Put_Col ("Racial Wealth", Creature.Wealth);
  else
    Put_Col ("Racial Wealth", "Average");
  end if;
  
  --------------------------------------------------------------------------------------------
  
  -- Close output file
  Ada.Text_Io.Close (File => Output_File);
  
  --------------------------------------------------------------------------------------------
  
end Create_Ur_Species;
